import { jsx, jsxs, Fragment } from 'react/jsx-runtime';
import { cx, css } from '@emotion/css';
import * as React from 'react';
import { selectors } from '@grafana/e2e-selectors';
import { useStyles2 } from '../../themes/ThemeContext.mjs';
import { getFocusStyles } from '../../themes/mixins.mjs';
import { clearButtonStyles } from '../Button/Button.mjs';
import { Icon } from '../Icon/Icon.mjs';
import { Tooltip } from '../Tooltip/Tooltip.mjs';
import { Counter } from './Counter.mjs';

const Tab = React.forwardRef(
  ({ label, active, icon, onChangeTab, counter, suffix: Suffix, className, href, truncate, tooltip, ...otherProps }, ref) => {
    const tabsStyles = useStyles2(getStyles);
    const clearStyles = useStyles2(clearButtonStyles);
    const content = () => /* @__PURE__ */ jsxs(Fragment, { children: [
      icon && /* @__PURE__ */ jsx(Icon, { name: icon, "data-testid": `tab-icon-${icon}` }),
      label,
      typeof counter === "number" && /* @__PURE__ */ jsx(Counter, { value: counter }),
      Suffix && /* @__PURE__ */ jsx(Suffix, { className: tabsStyles.suffix })
    ] });
    const linkClass = cx(
      clearStyles,
      tabsStyles.link,
      active ? tabsStyles.activeStyle : tabsStyles.notActive,
      truncate && tabsStyles.linkTruncate
    );
    const commonProps = {
      className: linkClass,
      "data-testid": selectors.components.Tab.title(label),
      ...otherProps,
      onClick: onChangeTab,
      role: "tab",
      "aria-selected": active,
      title: !!tooltip ? void 0 : otherProps.title
      // If tooltip is provided, don't set the title on the link or button, it looks weird
    };
    let tab = null;
    if (href) {
      tab = /* @__PURE__ */ jsx("div", { className: cx(tabsStyles.item, truncate && tabsStyles.itemTruncate, className), children: /* @__PURE__ */ jsx(
        "a",
        {
          ...commonProps,
          href,
          ref,
          children: content()
        }
      ) });
    } else {
      tab = /* @__PURE__ */ jsx("div", { className: cx(tabsStyles.item, truncate && tabsStyles.itemTruncate, className), children: /* @__PURE__ */ jsx(
        "button",
        {
          ...commonProps,
          type: "button",
          ref,
          children: content()
        }
      ) });
    }
    if (tooltip) {
      return /* @__PURE__ */ jsx(Tooltip, { content: tooltip, children: tab });
    }
    return tab;
  }
);
Tab.displayName = "Tab";
const getStyles = (theme) => {
  return {
    item: css({
      listStyle: "none",
      position: "relative",
      display: "flex",
      whiteSpace: "nowrap",
      padding: theme.spacing(0, 0.5)
    }),
    itemTruncate: css({
      maxWidth: theme.spacing(40)
    }),
    link: css({
      color: theme.colors.text.secondary,
      padding: theme.spacing(1, 1.5, 1),
      borderRadius: theme.shape.radius.default,
      display: "block",
      height: "100%",
      svg: {
        marginRight: theme.spacing(1)
      },
      "&:focus-visible": getFocusStyles(theme),
      "&::before": {
        display: "block",
        content: '" "',
        position: "absolute",
        left: 0,
        right: 0,
        height: "2px",
        borderRadius: theme.shape.radius.default,
        bottom: 0
      }
    }),
    linkTruncate: css({
      textOverflow: "ellipsis",
      whiteSpace: "nowrap",
      wordBreak: "break-word",
      overflow: "hidden"
    }),
    notActive: css({
      "a:hover, &:hover, &:focus": {
        color: theme.colors.text.primary,
        "&::before": {
          backgroundColor: theme.colors.action.hover
        }
      }
    }),
    activeStyle: css({
      label: "activeTabStyle",
      color: theme.colors.text.primary,
      overflow: "hidden",
      "&::before": {
        backgroundImage: theme.colors.gradients.brandHorizontal
      }
    }),
    suffix: css({
      marginLeft: theme.spacing(1)
    })
  };
};

export { Tab };
//# sourceMappingURL=Tab.mjs.map
