import { jsx } from 'react/jsx-runtime';
import { FieldType, isDataFrame, isTimeSeriesFrame } from '@grafana/data';
import { TableCellDisplayMode } from '@grafana/schema';
import { ActionsCell } from './ActionsCell.mjs';
import AutoCell from './AutoCell.mjs';
import { BarGaugeCell } from './BarGaugeCell.mjs';
import { DataLinksCell } from './DataLinksCell.mjs';
import { GeoCell } from './GeoCell.mjs';
import { ImageCell } from './ImageCell.mjs';
import { JSONCell } from './JSONCell.mjs';
import { PillCell } from './PillCell.mjs';
import { SparklineCell } from './SparklineCell.mjs';

const GAUGE_RENDERER = (props) => /* @__PURE__ */ jsx(
  BarGaugeCell,
  {
    field: props.field,
    value: props.value,
    theme: props.theme,
    height: props.height,
    width: props.width,
    rowIdx: props.rowIdx
  }
);
const AUTO_RENDERER = (props) => /* @__PURE__ */ jsx(
  AutoCell,
  {
    value: props.value,
    field: props.field,
    justifyContent: props.justifyContent,
    rowIdx: props.rowIdx,
    cellOptions: props.cellOptions
  }
);
const SPARKLINE_RENDERER = (props) => /* @__PURE__ */ jsx(
  SparklineCell,
  {
    value: props.value,
    field: props.field,
    justifyContent: props.justifyContent,
    timeRange: props.timeRange,
    rowIdx: props.rowIdx,
    theme: props.theme,
    width: props.width
  }
);
const JSON_RENDERER = (props) => /* @__PURE__ */ jsx(JSONCell, { justifyContent: props.justifyContent, value: props.value, field: props.field, rowIdx: props.rowIdx });
const GEO_RENDERER = (props) => /* @__PURE__ */ jsx(GeoCell, { value: props.value, justifyContent: props.justifyContent, height: props.height });
const IMAGE_RENDERER = (props) => /* @__PURE__ */ jsx(
  ImageCell,
  {
    cellOptions: props.cellOptions,
    field: props.field,
    height: props.height,
    justifyContent: props.justifyContent,
    value: props.value,
    rowIdx: props.rowIdx
  }
);
const DATA_LINKS_RENDERER = (props) => /* @__PURE__ */ jsx(DataLinksCell, { field: props.field, rowIdx: props.rowIdx });
const ACTIONS_RENDERER = ({ field, rowIdx, getActions = () => [] }) => /* @__PURE__ */ jsx(ActionsCell, { field, rowIdx, getActions });
const PILL_RENDERER = (props) => /* @__PURE__ */ jsx(PillCell, { ...props });
function isCustomCellOptions(options) {
  return options.type === TableCellDisplayMode.Custom;
}
const CUSTOM_RENDERER = (props) => {
  if (!isCustomCellOptions(props.cellOptions) || !props.cellOptions.cellComponent) {
    return null;
  }
  const CustomCellComponent = props.cellOptions.cellComponent;
  return /* @__PURE__ */ jsx(CustomCellComponent, { field: props.field, rowIndex: props.rowIdx, frame: props.frame, value: props.value });
};
const CELL_RENDERERS = {
  [TableCellDisplayMode.Sparkline]: SPARKLINE_RENDERER,
  [TableCellDisplayMode.Gauge]: GAUGE_RENDERER,
  [TableCellDisplayMode.JSONView]: JSON_RENDERER,
  [TableCellDisplayMode.Image]: IMAGE_RENDERER,
  [TableCellDisplayMode.DataLinks]: DATA_LINKS_RENDERER,
  [TableCellDisplayMode.Actions]: ACTIONS_RENDERER,
  [TableCellDisplayMode.Custom]: CUSTOM_RENDERER,
  [TableCellDisplayMode.ColorText]: AUTO_RENDERER,
  [TableCellDisplayMode.ColorBackground]: AUTO_RENDERER,
  [TableCellDisplayMode.Auto]: AUTO_RENDERER,
  [TableCellDisplayMode.Pill]: PILL_RENDERER
};
function getCellRenderer(field, cellOptions) {
  var _a;
  const cellType = (_a = cellOptions == null ? void 0 : cellOptions.type) != null ? _a : TableCellDisplayMode.Auto;
  if (cellType === TableCellDisplayMode.Auto) {
    return getAutoRendererResult(field);
  }
  return CELL_RENDERERS[cellType];
}
function getAutoRendererResult(field) {
  if (field.type === FieldType.geo) {
    return GEO_RENDERER;
  }
  if (field.type === FieldType.frame) {
    const firstValue = field.values[0];
    if (isDataFrame(firstValue) && isTimeSeriesFrame(firstValue)) {
      return SPARKLINE_RENDERER;
    } else {
      return JSON_RENDERER;
    }
  }
  if (field.type === FieldType.other) {
    return JSON_RENDERER;
  }
  return AUTO_RENDERER;
}

export { getAutoRendererResult, getCellRenderer };
//# sourceMappingURL=renderers.mjs.map
