import { jsxs, jsx, Fragment } from 'react/jsx-runtime';
import { WKT } from 'ol/format';
import { Geometry } from 'ol/geom';
import { FieldType } from '@grafana/data';
import { t } from '@grafana/i18n';
import { IconButton } from '../../../IconButton/IconButton.mjs';
import { TableCellInspectorMode } from '../../TableCellInspector.mjs';
import { TableCellDisplayMode } from '@grafana/schema';
import { FILTER_FOR_OPERATOR, FILTER_OUT_OPERATOR } from '../types.mjs';

function TableCellActions(props) {
  const {
    field,
    value,
    cellOptions,
    displayName,
    setIsInspecting,
    setContextMenuProps,
    onCellFilterAdded,
    className,
    cellInspect,
    showFilters
  } = props;
  return (
    // stopping propagation to prevent clicks within the actions menu from triggering the cell click events
    // for things like the data links tooltip.
    // eslint-disable-next-line jsx-a11y/click-events-have-key-events, jsx-a11y/no-static-element-interactions
    /* @__PURE__ */ jsxs("div", { className, onClick: (ev) => ev.stopPropagation(), children: [
      cellInspect && /* @__PURE__ */ jsx(
        IconButton,
        {
          name: "eye",
          "aria-label": t("grafana-ui.table.cell-inspect-tooltip", "Inspect value"),
          onClick: () => {
            let inspectValue = value;
            let mode = TableCellInspectorMode.text;
            if (field.type === FieldType.geo && value instanceof Geometry) {
              inspectValue = new WKT().writeGeometry(value, {
                featureProjection: "EPSG:3857",
                dataProjection: "EPSG:4326"
              });
              mode = TableCellInspectorMode.code;
            } else if ("cellType" in cellOptions && cellOptions.cellType === TableCellDisplayMode.JSONView) {
              mode = TableCellInspectorMode.code;
            }
            setContextMenuProps({
              value: String(inspectValue != null ? inspectValue : ""),
              mode
            });
            setIsInspecting(true);
          }
        }
      ),
      showFilters && /* @__PURE__ */ jsxs(Fragment, { children: [
        /* @__PURE__ */ jsx(
          IconButton,
          {
            name: "search-plus",
            "aria-label": t("grafana-ui.table.cell-filter-on", "Filter for value"),
            onClick: () => {
              onCellFilterAdded == null ? void 0 : onCellFilterAdded({
                key: displayName,
                operator: FILTER_FOR_OPERATOR,
                value: String(value != null ? value : "")
              });
            }
          }
        ),
        /* @__PURE__ */ jsx(
          IconButton,
          {
            name: "search-minus",
            "aria-label": t("grafana-ui.table.cell-filter-out", "Filter out value"),
            onClick: () => {
              onCellFilterAdded == null ? void 0 : onCellFilterAdded({
                key: displayName,
                operator: FILTER_OUT_OPERATOR,
                value: String(value != null ? value : "")
              });
            }
          }
        )
      ] })
    ] })
  );
}

export { TableCellActions };
//# sourceMappingURL=TableCellActions.mjs.map
