import { jsx } from 'react/jsx-runtime';
import { memo, useCallback, useMemo } from 'react';
import { FieldType, FieldMatcherID, fieldMatchers } from '@grafana/data';
import { t } from '@grafana/i18n';
import { getFieldTypeIconName } from '../../types/icon.mjs';
import { Select } from '../Select/Select.mjs';

const FieldTypeMatcherEditor = memo((props) => {
  const { data, options, onChange: onChangeFromProps, id } = props;
  const counts = useFieldCounts(data);
  const selectOptions = useSelectOptions(counts, options);
  const onChange = useCallback(
    (selection) => {
      return onChangeFromProps(selection.value);
    },
    [onChangeFromProps]
  );
  const selectedOption = selectOptions.find((v) => v.value === options);
  return /* @__PURE__ */ jsx(Select, { inputId: id, value: selectedOption, options: selectOptions, onChange });
});
FieldTypeMatcherEditor.displayName = "FieldTypeMatcherEditor";
const allFieldTypeIconOptions = [
  { value: FieldType.number, label: "Number", icon: getFieldTypeIconName(FieldType.number) },
  { value: FieldType.string, label: "String", icon: getFieldTypeIconName(FieldType.string) },
  { value: FieldType.time, label: "Time", icon: getFieldTypeIconName(FieldType.time) },
  { value: FieldType.boolean, label: "Boolean", icon: getFieldTypeIconName(FieldType.boolean) },
  { value: FieldType.trace, label: "Traces", icon: getFieldTypeIconName(FieldType.trace) },
  { value: FieldType.enum, label: "Enum", icon: getFieldTypeIconName(FieldType.enum) },
  { value: FieldType.other, label: "Other", icon: getFieldTypeIconName(FieldType.other) }
];
const useFieldCounts = (data) => {
  return useMemo(() => {
    const counts = /* @__PURE__ */ new Map();
    for (const t2 of allFieldTypeIconOptions) {
      counts.set(t2.value, 0);
    }
    for (const frame of data) {
      for (const field of frame.fields) {
        const key = field.type || FieldType.other;
        let v = counts.get(key);
        if (!v) {
          v = 0;
        }
        counts.set(key, v + 1);
      }
    }
    return counts;
  }, [data]);
};
const useSelectOptions = (counts, opt) => {
  return useMemo(() => {
    let found = false;
    const options = [];
    for (const t2 of allFieldTypeIconOptions) {
      const count = counts.get(t2.value);
      const match = opt === t2.value;
      if (count || match) {
        options.push({
          ...t2,
          label: `${t2.label} (${counts.get(t2.value)})`
        });
      }
      if (match) {
        found = true;
      }
    }
    if (opt && !found) {
      options.push({
        value: opt,
        label: `${opt} (No matches)`
      });
    }
    return options;
  }, [counts, opt]);
};
const getFieldTypeMatcherItem = () => ({
  id: FieldMatcherID.byType,
  component: FieldTypeMatcherEditor,
  matcher: fieldMatchers.get(FieldMatcherID.byType),
  name: t("grafana-ui.matchers-ui.name-fields-with-type", "Fields with type"),
  description: t(
    "grafana-ui.matchers-ui.description-fields-with-type",
    "Set properties for fields of a specific type (number, string, boolean)"
  ),
  optionsToLabel: (options) => options
});

export { FieldTypeMatcherEditor, allFieldTypeIconOptions, getFieldTypeMatcherItem };
//# sourceMappingURL=FieldTypeMatcherEditor.mjs.map
