import { clone } from 'lodash';
import { map } from 'rxjs/operators';
import { cacheFieldDisplayNames, getFieldDisplayName } from '../../field/fieldState.mjs';
import { FieldType } from '../../types/dataFrame.mjs';
import { DataTransformerID } from './ids.mjs';

const orderFieldsTransformer = {
  id: DataTransformerID.order,
  name: "Order fields by name",
  description: "Order fields based on configuration given by user",
  defaultOptions: {
    indexByName: {}
  },
  /**
   * Return a modified copy of the series. If the transform is not or should not
   * be applied, just return the input series
   */
  operator: ({ indexByName, orderByMode = "manual" /* Manual */, orderBy = [] }) => (source) => source.pipe(
    map((data) => {
      cacheFieldDisplayNames(data);
      const orderer = orderByMode === "manual" /* Manual */ ? createFieldsOrdererManual(indexByName) : createFieldsOrdererAuto(orderBy);
      return data.map((frame) => ({
        ...frame,
        fields: orderer(frame.fields, data, frame)
      }));
    })
  )
};
const createOrderFieldsComparer = (indexByName) => (a, b) => indexOfField(a, indexByName) - indexOfField(b, indexByName);
const createFieldsOrdererManual = (indexByName) => (fields, data, frame) => {
  const comparer = createOrderFieldsComparer(indexByName);
  return clone(fields).sort(
    (a, b) => comparer(getFieldDisplayName(a, frame, data), getFieldDisplayName(b, frame, data))
  );
};
const indexOfField = (fieldName, indexByName) => {
  if (Number.isInteger(indexByName[fieldName])) {
    return indexByName[fieldName];
  }
  return Number.MAX_SAFE_INTEGER;
};
const compare = new Intl.Collator(void 0, { sensitivity: "base", numeric: true }).compare;
const createFieldsOrdererAuto = (orderBy) => (fields) => {
  const firstTimeField = fields.find((f) => f.type === FieldType.time);
  return fields.slice().sort((fieldA, fieldB) => {
    var _a, _b, _c, _d, _e, _f, _g, _h;
    if (fieldA === firstTimeField) {
      return -1;
    }
    if (fieldB === firstTimeField) {
      return 1;
    }
    for (let i = 0; i < orderBy.length; i++) {
      let { type, name = "", desc = false } = orderBy[i];
      let aVal = type === "name" /* Name */ ? (_b = (_a = fieldA.state) == null ? void 0 : _a.displayName) != null ? _b : fieldA.name : (_d = (_c = fieldA.labels) == null ? void 0 : _c[name]) != null ? _d : "";
      let bVal = type === "name" /* Name */ ? (_f = (_e = fieldB.state) == null ? void 0 : _e.displayName) != null ? _f : fieldB.name : (_h = (_g = fieldB.labels) == null ? void 0 : _g[name]) != null ? _h : "";
      let res = compare(aVal, bVal) * (desc ? -1 : 1);
      if (res !== 0) {
        return res;
      }
    }
    return 0;
  });
};

export { createFieldsOrdererAuto, createOrderFieldsComparer, orderFieldsTransformer };
//# sourceMappingURL=order.mjs.map
