import { PrometheusCacheLevel } from './types';
/**
 * Returns the debounce time in milliseconds based on the cache level.
 * Used to control the frequency of API requests.
 *
 * @param {PrometheusCacheLevel} cacheLevel - The cache level (None, Low, Medium, High)
 * @returns {number} Debounce time in milliseconds:
 *   - Medium: 600ms
 *   - High: 1200ms
 *   - Default (None/Low): 350ms
 */
export declare const getDebounceTimeInMilliseconds: (cacheLevel: PrometheusCacheLevel) => number;
/**
 * Returns the number of days to cache metadata based on the cache level.
 * Used for caching Prometheus metric metadata.
 *
 * @param {PrometheusCacheLevel} cacheLevel - The cache level (None, Low, Medium, High)
 * @returns {number} Number of days to cache:
 *   - Medium: 7 days
 *   - High: 30 days
 *   - Default (None/Low): 1 day
 */
export declare const getDaysToCacheMetadata: (cacheLevel: PrometheusCacheLevel) => number;
/**
 * Returns the cache duration in minutes based on the cache level.
 * Used for general API response caching.
 *
 * @param {PrometheusCacheLevel} cacheLevel - The cache level (None, Low, Medium, High)
 * @returns Cache duration in minutes:
 *   - Medium: 10 minutes
 *   - High: 60 minutes
 *   - Default (None/Low): 1 minute
 */
export declare const getCacheDurationInMinutes: (cacheLevel: PrometheusCacheLevel) => 1 | 10 | 60;
/**
 * Builds cache headers for Prometheus API requests.
 * Creates a standard cache control header with private scope and max-age directive.
 *
 * @param {number} durationInSeconds - Cache duration in seconds
 * @returns Object containing headers with cache control directives:
 *   - X-Grafana-Cache: private, max-age=<duration>
 * @example
 * // Returns { headers: { 'X-Grafana-Cache': 'private, max-age=300' } }
 * buildCacheHeaders(300)
 */
export declare const buildCacheHeaders: (durationInSeconds: number) => {
    headers: {
        'X-Grafana-Cache': string;
    };
};
/**
 * Gets appropriate cache headers based on the configured cache level.
 * Converts cache duration from minutes to seconds and builds the headers.
 * Returns undefined if caching is disabled (None level).
 *
 * @param {PrometheusCacheLevel} cacheLevel - Cache level (None, Low, Medium, High)
 * @returns Cache headers object or undefined if caching is disabled
 * @example
 * // For Medium level, returns { headers: { 'X-Grafana-Cache': 'private, max-age=600' } }
 * getDefaultCacheHeaders(PrometheusCacheLevel.Medium)
 */
export declare const getDefaultCacheHeaders: (cacheLevel: PrometheusCacheLevel) => {
    headers: {
        'X-Grafana-Cache': string;
    };
} | undefined;
