import { jsx, jsxs } from 'react/jsx-runtime';
import { css } from '@emotion/css';
import { toOption } from '@grafana/data';
import { t } from '@grafana/i18n';
import { AutoSizeInput, Checkbox, useStyles2, Button, Stack, Select } from '@grafana/ui';
import { LabelParamEditor } from '../components/LabelParamEditor.mjs';
import { getOperationParamId } from './param_utils.mjs';

const editorMap = {
  // The wrapper component will ensure the modeller is provided
  LabelParamEditor
};
function getOperationParamEditor(paramDef) {
  if (paramDef.editor) {
    if (typeof paramDef.editor === "string") {
      return editorMap[paramDef.editor] || SimpleInputParamEditor;
    }
    return paramDef.editor;
  }
  if (paramDef.options) {
    return SelectInputParamEditor;
  }
  switch (paramDef.type) {
    case "boolean":
      return BoolInputParamEditor;
    case "number":
    case "string":
    default:
      return SimpleInputParamEditor;
  }
}
function SimpleInputParamEditor(props) {
  var _a;
  return /* @__PURE__ */ jsx(
    AutoSizeInput,
    {
      id: getOperationParamId(props.operationId, props.index),
      defaultValue: (_a = props.value) == null ? void 0 : _a.toString(),
      minWidth: props.paramDef.minWidth,
      placeholder: props.paramDef.placeholder,
      title: props.paramDef.description,
      maxWidth: (props.paramDef.minWidth || 20) * 3,
      onCommitChange: (evt) => {
        props.onChange(props.index, evt.currentTarget.value);
        if (props.paramDef.runQueryOnEnter && evt.type === "keydown") {
          props.onRunQuery();
        }
      }
    }
  );
}
function BoolInputParamEditor(props) {
  return /* @__PURE__ */ jsx(
    Checkbox,
    {
      id: getOperationParamId(props.operationId, props.index),
      value: Boolean(props.value),
      onChange: (evt) => props.onChange(props.index, evt.currentTarget.checked)
    }
  );
}
function SelectInputParamEditor({
  paramDef,
  value,
  index,
  operationId,
  onChange
}) {
  var _a, _b;
  const styles = useStyles2(getStyles);
  let selectOptions = paramDef.options;
  if (!((_a = selectOptions[0]) == null ? void 0 : _a.label)) {
    selectOptions = paramDef.options.map((option) => ({
      label: option.toString(),
      value: option
    }));
  }
  let valueOption = (_b = selectOptions.find((x) => x.value === value)) != null ? _b : toOption(value);
  if (!value && paramDef.optional) {
    return /* @__PURE__ */ jsx("div", { className: styles.optionalParam, children: /* @__PURE__ */ jsx(
      Button,
      {
        size: "sm",
        variant: "secondary",
        title: t("grafana-prometheus.querybuilder.operation-param-editor.title-add", "Add {{name}}", {
          name: paramDef.name
        }),
        icon: "plus",
        onClick: () => onChange(index, selectOptions[0].value),
        children: paramDef.name
      }
    ) });
  }
  return /* @__PURE__ */ jsxs(Stack, { gap: 0.5, direction: "row", alignItems: "center", children: [
    /* @__PURE__ */ jsx(
      Select,
      {
        id: getOperationParamId(operationId, index),
        value: valueOption,
        options: selectOptions,
        placeholder: paramDef.placeholder,
        allowCustomValue: true,
        onChange: (value2) => onChange(index, value2.value),
        width: paramDef.minWidth || "auto"
      }
    ),
    paramDef.optional && /* @__PURE__ */ jsx(
      Button,
      {
        "data-testid": `operations.${index}.remove-param`,
        size: "sm",
        fill: "text",
        icon: "times",
        variant: "secondary",
        title: t("grafana-prometheus.querybuilder.operation-param-editor.title-remove", "Remove {{name}}", {
          name: paramDef.name
        }),
        onClick: () => onChange(index, "")
      }
    )
  ] });
}
const getStyles = (theme) => {
  return {
    optionalParam: css({
      marginTop: theme.spacing(1)
    })
  };
};

export { getOperationParamEditor };
//# sourceMappingURL=OperationParamEditorRegistry.mjs.map
