import { jsx, jsxs } from 'react/jsx-runtime';
import { cx, css } from '@emotion/css';
import { Draggable } from '@hello-pangea/dnd';
import { useId, useState, useEffect } from 'react';
import { Trans, t } from '@grafana/i18n';
import { useStyles2, Tooltip, Icon, Stack, Button } from '@grafana/ui';
import { OperationHeader } from './OperationHeader.mjs';
import { getOperationParamEditor } from './OperationParamEditorRegistry.mjs';
import { getOperationParamId } from './param_utils.mjs';

function OperationEditor({
  operation,
  index,
  onRemove,
  onChange,
  onRunQuery,
  queryModeller,
  query,
  datasource,
  flash,
  highlight,
  timeRange
}) {
  const styles = useStyles2(getStyles);
  const def = queryModeller.getOperationDef(operation.id);
  const shouldFlash = useFlash(flash);
  const id = useId();
  if (!def) {
    return /* @__PURE__ */ jsx("span", { children: /* @__PURE__ */ jsxs(Trans, { i18nKey: "grafana-prometheus.querybuilder.operation-editor.not-found", values: { id: operation.id }, children: [
      "Operation ",
      "{{id}}",
      " not found"
    ] }) });
  }
  const onParamValueChanged = (paramIdx, value) => {
    const update = { ...operation, params: [...operation.params] };
    update.params[paramIdx] = value;
    callParamChangedThenOnChange(def, update, index, paramIdx, onChange);
  };
  const onAddRestParam = () => {
    const update = { ...operation, params: [...operation.params, ""] };
    callParamChangedThenOnChange(def, update, index, operation.params.length, onChange);
  };
  const onRemoveRestParam = (paramIdx) => {
    const update = {
      ...operation,
      params: [...operation.params.slice(0, paramIdx), ...operation.params.slice(paramIdx + 1)]
    };
    callParamChangedThenOnChange(def, update, index, paramIdx, onChange);
  };
  const operationElements = [];
  for (let paramIndex = 0; paramIndex < operation.params.length; paramIndex++) {
    const paramDef = def.params[Math.min(def.params.length - 1, paramIndex)];
    const Editor = getOperationParamEditor(paramDef);
    operationElements.push(
      /* @__PURE__ */ jsxs("div", { className: styles.paramRow, children: [
        !paramDef.hideName && /* @__PURE__ */ jsxs("div", { className: styles.paramName, children: [
          /* @__PURE__ */ jsx("label", { htmlFor: getOperationParamId(id, paramIndex), children: paramDef.name }),
          paramDef.description && /* @__PURE__ */ jsx(Tooltip, { placement: "top", content: paramDef.description, theme: "info", children: /* @__PURE__ */ jsx(Icon, { name: "info-circle", size: "sm", className: styles.infoIcon }) })
        ] }),
        /* @__PURE__ */ jsx("div", { className: styles.paramValue, children: /* @__PURE__ */ jsxs(Stack, { gap: 0.5, direction: "row", alignItems: "center", children: [
          /* @__PURE__ */ jsx(
            Editor,
            {
              paramDef,
              value: operation.params[paramIndex],
              index: paramIndex,
              operationId: operation.id,
              query,
              datasource,
              timeRange,
              onChange: onParamValueChanged,
              onRunQuery,
              queryModeller
            }
          ),
          paramDef.restParam && (operation.params.length > def.params.length || paramDef.optional) && /* @__PURE__ */ jsx(
            Button,
            {
              "data-testid": `operations.${index}.remove-rest-param`,
              size: "sm",
              fill: "text",
              icon: "times",
              variant: "secondary",
              title: t("grafana-prometheus.querybuilder.operation-editor.title-remove", "Remove {{name}}", {
                name: paramDef.name
              }),
              onClick: () => onRemoveRestParam(paramIndex)
            }
          )
        ] }) })
      ] }, `${paramIndex}-1`)
    );
  }
  let restParam;
  if (def.params.length > 0) {
    const lastParamDef = def.params[def.params.length - 1];
    if (lastParamDef.restParam) {
      restParam = renderAddRestParamButton(lastParamDef, onAddRestParam, index, operation.params.length, styles);
    }
  }
  return /* @__PURE__ */ jsx(Draggable, { draggableId: `operation-${index}`, index, children: (provided) => /* @__PURE__ */ jsxs(
    "div",
    {
      className: cx(styles.card, (shouldFlash || highlight) && styles.cardHighlight),
      ref: provided.innerRef,
      ...provided.draggableProps,
      "data-testid": `operations.${index}.wrapper`,
      children: [
        /* @__PURE__ */ jsx(
          OperationHeader,
          {
            operation,
            dragHandleProps: provided.dragHandleProps,
            def,
            index,
            onChange,
            onRemove,
            queryModeller
          }
        ),
        /* @__PURE__ */ jsx("div", { className: styles.body, children: operationElements }),
        restParam,
        index < query.operations.length - 1 && /* @__PURE__ */ jsxs("div", { className: styles.arrow, children: [
          /* @__PURE__ */ jsx("div", { className: styles.arrowLine }),
          /* @__PURE__ */ jsx("div", { className: styles.arrowArrow })
        ] })
      ]
    }
  ) });
}
function useFlash(flash) {
  const [keepFlash, setKeepFlash] = useState(true);
  useEffect(() => {
    let t2;
    if (flash) {
      t2 = setTimeout(() => {
        setKeepFlash(false);
      }, 1e3);
    } else {
      setKeepFlash(true);
    }
    return () => clearTimeout(t2);
  }, [flash]);
  return keepFlash && flash;
}
function renderAddRestParamButton(paramDef, onAddRestParam, operationIndex, paramIndex, styles) {
  return /* @__PURE__ */ jsx("div", { className: styles.restParam, children: /* @__PURE__ */ jsx(
    Button,
    {
      size: "sm",
      icon: "plus",
      title: `Add ${paramDef.name}`.trimEnd(),
      variant: "secondary",
      onClick: onAddRestParam,
      "data-testid": `operations.${operationIndex}.add-rest-param`,
      children: paramDef.name
    }
  ) }, `${paramIndex}-2`);
}
function callParamChangedThenOnChange(def, operation, operationIndex, paramIndex, onChange) {
  if (def.paramChangedHandler) {
    onChange(operationIndex, def.paramChangedHandler(paramIndex, operation, def));
  } else {
    onChange(operationIndex, operation);
  }
}
const getStyles = (theme) => {
  return {
    cardWrapper: css({
      alignItems: "stretch"
    }),
    error: css({
      marginBottom: theme.spacing(1)
    }),
    card: css({
      background: theme.colors.background.primary,
      border: `1px solid ${theme.colors.border.medium}`,
      cursor: "grab",
      borderRadius: theme.shape.radius.default,
      marginBottom: theme.spacing(1),
      position: "relative",
      [theme.transitions.handleMotion("no-preference", "reduce")]: {
        transition: "all 0.5s ease-in 0s"
      },
      height: "100%"
    }),
    cardError: css({
      boxShadow: `0px 0px 4px 0px ${theme.colors.warning.main}`,
      border: `1px solid ${theme.colors.warning.main}`
    }),
    cardHighlight: css({
      boxShadow: `0px 0px 4px 0px ${theme.colors.primary.border}`,
      border: `1px solid ${theme.colors.primary.border}`
    }),
    infoIcon: css({
      marginLeft: theme.spacing(0.5),
      color: theme.colors.text.secondary,
      ":hover": {
        color: theme.colors.text.primary
      }
    }),
    body: css({
      margin: theme.spacing(1, 1, 0.5, 1),
      display: "table"
    }),
    paramRow: css({
      label: "paramRow",
      display: "table-row",
      verticalAlign: "middle"
    }),
    paramName: css({
      display: "table-cell",
      padding: theme.spacing(0, 1, 0, 0),
      fontSize: theme.typography.bodySmall.fontSize,
      fontWeight: theme.typography.fontWeightMedium,
      verticalAlign: "middle",
      height: "32px"
    }),
    paramValue: css({
      label: "paramValue",
      display: "table-cell",
      verticalAlign: "middle"
    }),
    restParam: css({
      padding: theme.spacing(0, 1, 1, 1)
    }),
    arrow: css({
      position: "absolute",
      top: "0",
      right: "-18px",
      display: "flex"
    }),
    arrowLine: css({
      height: "2px",
      width: "8px",
      backgroundColor: theme.colors.border.strong,
      position: "relative",
      top: "14px"
    }),
    arrowArrow: css({
      width: 0,
      height: 0,
      borderTop: `5px solid transparent`,
      borderBottom: `5px solid transparent`,
      borderLeft: `7px solid ${theme.colors.border.strong}`,
      position: "relative",
      top: "10px"
    })
  };
};

export { OperationEditor };
//# sourceMappingURL=OperationEditor.mjs.map
