import { defaults } from 'lodash';
import { tz } from 'moment-timezone';
import { throwError, lastValueFrom } from 'rxjs';
import { map, tap } from 'rxjs/operators';
import { gte } from 'semver';
import { dateTime, getDefaultTimeRange, rangeUtil, scopeFilterOperatorMap, CoreApp } from '@grafana/data';
import { DataSourceWithBackend, getTemplateSrv, getBackendSrv, isFetchError, config } from '@grafana/runtime';
import { addLabelToQuery } from './add_label_to_query.mjs';
import { PrometheusAnnotationSupport } from './annotations.mjs';
import { SUGGESTIONS_LIMIT, DEFAULT_SERIES_LIMIT, GET_AND_POST_METADATA_ENDPOINTS, InstantQueryRefIdIndex } from './constants.mjs';
import { prometheusRegularEscape, prometheusSpecialRegexEscape } from './escaping.mjs';
import { PrometheusLanguageProvider, importFromAbstractQuery, exportToAbstractQuery, populateMatchParamsFromQueries } from './language_provider.mjs';
import { expandRecordingRules, getRangeSnapInterval, getPrometheusTime } from './language_utils.mjs';
import { PrometheusMetricFindQuery } from './metric_find_query.mjs';
import { getQueryHints } from './query_hints.mjs';
import { renderLabelsWithoutBrackets } from './querybuilder/shared/rendering/labels.mjs';
import { QueryCache, defaultPrometheusQueryOverlapWindow } from './querycache/QueryCache.mjs';
import { transformV2 } from './result_transformer.mjs';
import { trackQuery } from './tracking.mjs';
import { PrometheusCacheLevel, PromApplication } from './types.mjs';
import { wrapUtf8Filters, utf8Support } from './utf8_support.mjs';
import { PrometheusVariableSupport } from './variables.mjs';

class PrometheusDatasource extends DataSourceWithBackend {
  constructor(instanceSettings, templateSrv = getTemplateSrv(), languageProvider) {
    var _a, _b, _c, _d, _e, _f, _g, _h;
    super(instanceSettings);
    this.templateSrv = templateSrv;
    /**
     * Initializes the Prometheus datasource by loading recording rules and checking exemplar availability.
     *
     * This method performs two key initialization tasks: Loads recording rules from the
     * Prometheus API and checks if exemplars are available by testing the exemplars API endpoint.
     */
    this.init = async () => {
      if (!this.disableRecordingRules) {
        this.loadRules();
      }
      this.exemplarsAvailable = await this.areExemplarsAvailable();
    };
    this.access = instanceSettings.access;
    this.basicAuth = instanceSettings.basicAuth;
    this.cache = new QueryCache({
      getTargetSignature: this.getPrometheusTargetSignature.bind(this),
      overlapString: (_a = instanceSettings.jsonData.incrementalQueryOverlapWindow) != null ? _a : defaultPrometheusQueryOverlapWindow,
      applyInterpolation: this.interpolateString.bind(this)
    });
    this.cacheLevel = (_b = instanceSettings.jsonData.cacheLevel) != null ? _b : PrometheusCacheLevel.Low;
    this.customQueryParameters = new URLSearchParams(instanceSettings.jsonData.customQueryParameters);
    this.datasourceConfigurationPrometheusFlavor = instanceSettings.jsonData.prometheusType;
    this.datasourceConfigurationPrometheusVersion = instanceSettings.jsonData.prometheusVersion;
    this.disableRecordingRules = (_c = instanceSettings.jsonData.disableRecordingRules) != null ? _c : false;
    this.exemplarTraceIdDestinations = instanceSettings.jsonData.exemplarTraceIdDestinations;
    this.exemplarsAvailable = true;
    this.hasIncrementalQuery = (_d = instanceSettings.jsonData.incrementalQuerying) != null ? _d : false;
    this.httpMethod = instanceSettings.jsonData.httpMethod || "GET";
    this.id = instanceSettings.id;
    this.interval = instanceSettings.jsonData.timeInterval || "15s";
    this.lookupsDisabled = (_e = instanceSettings.jsonData.disableMetricsLookup) != null ? _e : false;
    this.metricNamesAutocompleteSuggestionLimit = (_f = instanceSettings.jsonData.codeModeMetricNamesSuggestionLimit) != null ? _f : SUGGESTIONS_LIMIT;
    this.ruleMappings = {};
    this.seriesEndpoint = (_g = instanceSettings.jsonData.seriesEndpoint) != null ? _g : false;
    this.seriesLimit = (_h = instanceSettings.jsonData.seriesLimit) != null ? _h : DEFAULT_SERIES_LIMIT;
    this.type = "prometheus";
    this.url = instanceSettings.url;
    this.withCredentials = Boolean(instanceSettings.withCredentials);
    this.defaultEditor = instanceSettings.jsonData.defaultEditor;
    this.annotations = PrometheusAnnotationSupport();
    this.variables = new PrometheusVariableSupport(this, this.templateSrv);
    this.languageProvider = languageProvider != null ? languageProvider : new PrometheusLanguageProvider(this);
  }
  /**
   * Loads recording rules from the Prometheus API and extracts rule mappings.
   *
   * This method fetches rules from the `/api/v1/rules` endpoint and processes
   * them to create a mapping of rule names to their corresponding queries and labels.
   * The rules API is experimental, so errors are logged but not thrown.
   */
  async loadRules() {
    var _a, _b;
    try {
      const params = {};
      const options = { showErrorAlert: false };
      const res = await this.metadataRequest("/api/v1/rules", params, options);
      const ruleGroups = (_b = (_a = res.data) == null ? void 0 : _a.data) == null ? void 0 : _b.groups;
      if (ruleGroups) {
        this.ruleMappings = extractRuleMappingFromGroups(ruleGroups);
      }
    } catch (err) {
      console.log("Rules API is experimental. Ignore next error.");
      console.error(err);
    }
  }
  /**
   * Checks if exemplars are available by testing the exemplars API endpoint.
   *
   * This method makes a test request to the `/api/v1/query_exemplars` endpoint to determine
   * if the Prometheus instance supports exemplars. The test uses a simple query with a
   * 30-minute time range. If the request succeeds with a 'success' status, exemplars
   * are considered available. Errors are caught and return false to avoid breaking
   * the datasource initialization.
   */
  async areExemplarsAvailable() {
    try {
      const params = {
        query: "test",
        start: dateTime().subtract(30, "minutes").valueOf().toString(),
        end: dateTime().valueOf().toString()
      };
      const options = { showErrorAlert: false };
      const res = await this.metadataRequest("/api/v1/query_exemplars", params, options);
      return res.data.status === "success";
    } catch (err) {
      return false;
    }
  }
  getQueryDisplayText(query) {
    return query.expr;
  }
  /**
   * Get target signature for query caching
   * @param request
   * @param query
   */
  getPrometheusTargetSignature(request, query) {
    var _a, _b;
    const targExpr = this.interpolateString(query.expr);
    return `${targExpr}|${(_a = query.interval) != null ? _a : request.interval}|${JSON.stringify((_b = request.rangeRaw) != null ? _b : "")}|${query.exemplar}`;
  }
  hasLabelsMatchAPISupport() {
    if (this.seriesEndpoint) {
      return false;
    }
    return (
      // https://github.com/prometheus/prometheus/releases/tag/v2.24.0
      this._isDatasourceVersionGreaterOrEqualTo("2.24.0", PromApplication.Prometheus) || // All versions of Mimir support matchers for labels API
      this._isDatasourceVersionGreaterOrEqualTo("2.0.0", PromApplication.Mimir) || // https://github.com/cortexproject/cortex/discussions/4542
      this._isDatasourceVersionGreaterOrEqualTo("1.11.0", PromApplication.Cortex) || // https://github.com/thanos-io/thanos/pull/3566
      //https://github.com/thanos-io/thanos/releases/tag/v0.18.0
      this._isDatasourceVersionGreaterOrEqualTo("0.18.0", PromApplication.Thanos)
    );
  }
  _isDatasourceVersionGreaterOrEqualTo(targetVersion, targetFlavor) {
    if (!this.datasourceConfigurationPrometheusVersion || !this.datasourceConfigurationPrometheusFlavor) {
      return true;
    }
    if (targetFlavor !== this.datasourceConfigurationPrometheusFlavor) {
      return false;
    }
    return gte(this.datasourceConfigurationPrometheusVersion, targetVersion);
  }
  _addTracingHeaders(httpOptions, options) {
    httpOptions.headers = {};
    if (this.access === "proxy") {
      httpOptions.headers["X-Dashboard-UID"] = options.dashboardUID;
      httpOptions.headers["X-Panel-Id"] = options.panelId;
    }
  }
  directAccessError() {
    return throwError(
      () => new Error(
        "Browser access mode in the Prometheus datasource is no longer available. Switch to server access mode."
      )
    );
  }
  /**
   * Any request done from this data source should go through here as it contains some common processing for the
   * request. Any processing done here needs to be also copied on the backend as this goes through data source proxy
   * but not through the same code as alerting.
   */
  _request(url, data, overrides = {}) {
    if (this.access === "direct") {
      return this.directAccessError();
    }
    data = data || {};
    for (const [key, value] of this.customQueryParameters) {
      if (data[key] == null) {
        data[key] = value;
      }
    }
    let queryUrl = this.url + url;
    if (url.startsWith(`/api/datasources/uid/${this.uid}`)) {
      queryUrl = url;
    }
    const options = defaults(overrides, {
      url: queryUrl,
      method: this.httpMethod,
      headers: {}
    });
    if (options.method === "GET") {
      if (data && Object.keys(data).length) {
        options.url = options.url + (options.url.search(/\?/) >= 0 ? "&" : "?") + Object.entries(data).map(([k, v]) => `${encodeURIComponent(k)}=${encodeURIComponent(v)}`).join("&");
      }
    } else {
      if (!options.headers["Content-Type"]) {
        options.headers["Content-Type"] = "application/x-www-form-urlencoded";
      }
      options.data = data;
    }
    if (this.basicAuth || this.withCredentials) {
      options.withCredentials = true;
    }
    if (this.basicAuth) {
      options.headers.Authorization = this.basicAuth;
    }
    return getBackendSrv().fetch(options);
  }
  async importFromAbstractQueries(abstractQueries) {
    return abstractQueries.map((abstractQuery) => importFromAbstractQuery(abstractQuery));
  }
  async exportToAbstractQueries(queries) {
    return queries.map((query) => exportToAbstractQuery(query));
  }
  // Use this for tab completion features, wont publish response to other components
  async metadataRequest(url, params = {}, options) {
    if (GET_AND_POST_METADATA_ENDPOINTS.some((endpoint) => url.includes(endpoint))) {
      try {
        return await lastValueFrom(
          this._request(`/api/datasources/uid/${this.uid}/resources${url}`, params, {
            method: this.httpMethod,
            hideFromInspector: true,
            showErrorAlert: false,
            ...options
          })
        );
      } catch (err) {
        if (this.httpMethod === "POST" && isFetchError(err) && (err.status === 405 || err.status === 400)) {
          console.warn(`Couldn't use configured POST HTTP method for this request. Trying to use GET method instead.`);
        } else {
          throw err;
        }
      }
    }
    return await lastValueFrom(
      this._request(`/api/datasources/uid/${this.uid}/resources${url}`, params, {
        method: "GET",
        hideFromInspector: true,
        ...options
      })
    );
  }
  interpolateQueryExpr(value = [], variable) {
    if (!variable.multi && !variable.includeAll) {
      return prometheusRegularEscape(value);
    }
    if (typeof value === "string") {
      return prometheusSpecialRegexEscape(value);
    }
    const escapedValues = value.map((val) => prometheusSpecialRegexEscape(val));
    if (escapedValues.length === 1) {
      return escapedValues[0];
    }
    return "(" + escapedValues.join("|") + ")";
  }
  targetContainsTemplate(target) {
    return this.templateSrv.containsTemplate(target.expr);
  }
  shouldRunExemplarQuery(target, request) {
    if (target.exemplar) {
      const metricName = this.languageProvider.retrieveHistogramMetrics().find((m) => target.expr.includes(m));
      const currentTargetIdx = request.targets.findIndex((t) => t.refId === target.refId);
      const targets = request.targets.slice(0, currentTargetIdx).filter((t) => !t.hide);
      if (!metricName || metricName && !targets.some((t) => t.expr.includes(metricName))) {
        return true;
      }
      return false;
    }
    return false;
  }
  processTargetV2(target, request) {
    var _a;
    let utcOffset = request.range.to.utcOffset();
    if (request.timezone === "browser") {
      utcOffset = this.isUsingRelativeTimeRange(request.range) ? utcOffset : 0;
    } else {
      utcOffset = tz(request.timezone).utcOffset();
    }
    const processedTargets = [];
    const processedTarget = {
      ...target,
      exemplar: this.shouldRunExemplarQuery(target, request),
      requestId: request.panelId + target.refId,
      utcOffsetSec: utcOffset * 60
    };
    if (config.featureToggles.promQLScope) {
      processedTarget.scopes = ((_a = request.scopes) != null ? _a : []).map((scope) => ({
        name: scope.metadata.name,
        ...scope.spec
      }));
    }
    if (config.featureToggles.groupByVariable) {
      processedTarget.groupByKeys = request.groupByKeys;
    }
    if (target.instant && target.range) {
      processedTargets.push(
        {
          ...processedTarget,
          refId: processedTarget.refId,
          instant: false
        },
        {
          ...processedTarget,
          refId: processedTarget.refId + InstantQueryRefIdIndex,
          range: false,
          exemplar: false
        }
      );
    } else {
      processedTargets.push(processedTarget);
    }
    return processedTargets;
  }
  query(request) {
    if (this.access === "direct") {
      return this.directAccessError();
    }
    let fullOrPartialRequest;
    let requestInfo = void 0;
    const hasInstantQuery = request.targets.some((target) => target.instant);
    if (this.hasIncrementalQuery && !hasInstantQuery) {
      requestInfo = this.cache.requestInfo(request);
      fullOrPartialRequest = requestInfo.requests[0];
    } else {
      fullOrPartialRequest = request;
    }
    const targets = fullOrPartialRequest.targets.map((target) => this.processTargetV2(target, fullOrPartialRequest));
    const startTime = /* @__PURE__ */ new Date();
    return super.query({ ...fullOrPartialRequest, targets: targets.flat() }).pipe(
      map((response) => {
        const amendedResponse = {
          ...response,
          data: this.cache.procFrames(request, requestInfo, response.data)
        };
        return transformV2(amendedResponse, request, {
          exemplarTraceIdDestinations: this.exemplarTraceIdDestinations
        });
      }),
      tap((response) => {
        trackQuery(response, request, startTime);
      })
    );
  }
  metricFindQuery(query, options) {
    var _a;
    if (!query) {
      return Promise.resolve([]);
    }
    const timeRange = (_a = options == null ? void 0 : options.range) != null ? _a : getDefaultTimeRange();
    const scopedVars = {
      ...this.getIntervalVars(),
      ...this.getRangeScopedVars(timeRange)
    };
    const interpolated = this.templateSrv.replace(query, scopedVars, this.interpolateQueryExpr);
    const metricFindQuery = new PrometheusMetricFindQuery(this, interpolated);
    return metricFindQuery.process(timeRange);
  }
  getIntervalVars() {
    return {
      __interval: { text: this.interval, value: this.interval },
      __interval_ms: { text: rangeUtil.intervalToMs(this.interval), value: rangeUtil.intervalToMs(this.interval) }
    };
  }
  getRangeScopedVars(range) {
    const msRange = range.to.diff(range.from);
    const sRange = Math.round(msRange / 1e3);
    return {
      __range_ms: { text: msRange, value: msRange },
      __range_s: { text: sRange, value: sRange },
      __range: { text: sRange + "s", value: sRange + "s" }
    };
  }
  // By implementing getTagKeys and getTagValues we add ad-hoc filters functionality
  // this is used to get label keys, a.k.a label names
  // it is used in metric_find_query.ts
  // and in Tempo here grafana/public/app/plugins/datasource/tempo/QueryEditor/ServiceGraphSection.tsx
  async getTagKeys(options) {
    var _a, _b, _c;
    if (!options.timeRange) {
      options.timeRange = getDefaultTimeRange();
    }
    if (config.featureToggles.promQLScope && ((_b = (_a = options == null ? void 0 : options.scopes) == null ? void 0 : _a.length) != null ? _b : 0) > 0) {
      const suggestions = await this.languageProvider.fetchSuggestions(
        options.timeRange,
        options.queries,
        options.scopes,
        options.filters
      );
      return suggestions.filter((labelName) => !!labelName && !options.filters.find((filter) => filter.key === labelName)).map((k) => ({ value: k, text: k }));
    }
    const match = extractResourceMatcher((_c = options.queries) != null ? _c : [], options.filters);
    let labelKeys = await this.languageProvider.queryLabelKeys(options.timeRange, match);
    return labelKeys.filter((labelName) => !options.filters.find((filter) => filter.key === labelName)).map((k) => ({ value: k, text: k }));
  }
  // By implementing getTagKeys and getTagValues we add ad-hoc filters functionality
  async getTagValues(options) {
    var _a, _b, _c;
    if (!options.timeRange) {
      options.timeRange = getDefaultTimeRange();
    }
    const requestId = `[${this.uid}][${options.key}]`;
    if (config.featureToggles.promQLScope && ((_b = (_a = options == null ? void 0 : options.scopes) == null ? void 0 : _a.length) != null ? _b : 0) > 0) {
      return (await this.languageProvider.fetchSuggestions(
        options.timeRange,
        options.queries,
        options.scopes,
        options.filters,
        options.key,
        void 0,
        requestId
      )).map((v) => ({ value: v, text: v }));
    }
    const match = extractResourceMatcher((_c = options.queries) != null ? _c : [], options.filters);
    return (await this.languageProvider.queryLabelValues(options.timeRange, options.key, match)).map((v) => ({
      value: v,
      text: v
    }));
  }
  interpolateVariablesInQueries(queries, scopedVars, filters) {
    let expandedQueries = queries;
    if (queries && queries.length) {
      expandedQueries = queries.map((query) => {
        const interpolatedQuery = this.templateSrv.replace(
          query.expr,
          scopedVars,
          this.interpolateExploreMetrics(query.fromExploreMetrics)
        );
        const replacedInterpolatedQuery = config.featureToggles.promQLScope ? interpolatedQuery : this.templateSrv.replace(
          this.enhanceExprWithAdHocFilters(filters, interpolatedQuery),
          scopedVars,
          this.interpolateQueryExpr
        );
        const expandedQuery = {
          ...query,
          ...config.featureToggles.promQLScope ? { adhocFilters: this.generateScopeFilters(filters) } : {},
          datasource: this.getRef(),
          expr: replacedInterpolatedQuery,
          interval: this.templateSrv.replace(query.interval, scopedVars)
        };
        return expandedQuery;
      });
    }
    return expandedQueries;
  }
  getQueryHints(query, result) {
    var _a;
    return getQueryHints((_a = query.expr) != null ? _a : "", result, this);
  }
  modifyQuery(query, action) {
    var _a, _b, _c;
    let expression = (_a = query.expr) != null ? _a : "";
    switch (action.type) {
      case "ADD_FILTER": {
        const { key, value } = (_b = action.options) != null ? _b : {};
        if (key && value) {
          expression = addLabelToQuery(expression, key, value);
        }
        break;
      }
      case "ADD_FILTER_OUT": {
        const { key, value } = (_c = action.options) != null ? _c : {};
        if (key && value) {
          expression = addLabelToQuery(expression, key, value, "!=");
        }
        break;
      }
      case "ADD_HISTOGRAM_QUANTILE": {
        expression = `histogram_quantile(0.95, sum(rate(${expression}[$__rate_interval])) by (le))`;
        break;
      }
      case "ADD_HISTOGRAM_AVG": {
        expression = `histogram_avg(rate(${expression}[$__rate_interval]))`;
        break;
      }
      case "ADD_HISTOGRAM_FRACTION": {
        expression = `histogram_fraction(0,0.2,rate(${expression}[$__rate_interval]))`;
        break;
      }
      case "ADD_HISTOGRAM_COUNT": {
        expression = `histogram_count(rate(${expression}[$__rate_interval]))`;
        break;
      }
      case "ADD_HISTOGRAM_SUM": {
        expression = `histogram_sum(rate(${expression}[$__rate_interval]))`;
        break;
      }
      case "ADD_HISTOGRAM_STDDEV": {
        expression = `histogram_stddev(rate(${expression}[$__rate_interval]))`;
        break;
      }
      case "ADD_HISTOGRAM_STDVAR": {
        expression = `histogram_stdvar(rate(${expression}[$__rate_interval]))`;
        break;
      }
      case "ADD_RATE": {
        expression = `rate(${expression}[$__rate_interval])`;
        break;
      }
      case "ADD_SUM": {
        expression = `sum(${expression.trim()}) by ($1)`;
        break;
      }
      case "EXPAND_RULES": {
        if (action.options) {
          expression = expandRecordingRules(expression, action.options);
        }
        break;
      }
    }
    return { ...query, expr: expression };
  }
  /**
   * Returns the adjusted "snapped" interval parameters
   */
  getAdjustedInterval(timeRange) {
    return getRangeSnapInterval(this.cacheLevel, timeRange);
  }
  /**
   * This will return a time range that always includes the users current time range,
   * and then a little extra padding to round up/down to the nearest nth minute,
   * defined by the result of the getCacheDurationInMinutes.
   *
   * For longer cache durations, and shorter query durations,
   * the window we're calculating might be much bigger then the user's current window,
   * resulting in us returning labels/values that might not be applicable for the given window,
   * this is a necessary trade-off if we want to cache larger durations
   */
  getTimeRangeParams(timeRange) {
    return {
      start: getPrometheusTime(timeRange.from, false).toString(),
      end: getPrometheusTime(timeRange.to, true).toString()
    };
  }
  /**
   * This converts the adhocVariableFilter array and converts it to scopeFilter array
   * @param filters
   */
  generateScopeFilters(filters) {
    if (!filters) {
      return [];
    }
    return filters.map((f) => {
      var _a;
      return {
        key: f.key,
        operator: scopeFilterOperatorMap[f.operator],
        value: this.templateSrv.replace(f.value, {}, this.interpolateQueryExpr),
        values: (_a = f.values) == null ? void 0 : _a.map((v) => this.templateSrv.replace(v, {}, this.interpolateQueryExpr))
      };
    });
  }
  enhanceExprWithAdHocFilters(filters, expr) {
    if (!filters || filters.length === 0) {
      return expr;
    }
    const finalQuery = filters.reduce((acc, filter) => {
      const { key, operator } = filter;
      let { value } = filter;
      if (operator === "=~" || operator === "!~") {
        value = prometheusRegularEscape(value);
      }
      return addLabelToQuery(acc, key, value, operator);
    }, expr);
    return finalQuery;
  }
  // Used when running queries through backend
  filterQuery(query) {
    if (query.hide || !query.expr) {
      return false;
    }
    return true;
  }
  // Used when running queries through backend
  applyTemplateVariables(target, scopedVars, filters) {
    const variables = { ...scopedVars };
    variables.__interval = {
      value: "$__interval"
    };
    variables.__interval_ms = {
      value: "$__interval_ms"
    };
    const expr = this.templateSrv.replace(
      target.expr,
      variables,
      this.interpolateExploreMetrics(target.fromExploreMetrics)
    );
    const exprWithAdhoc = config.featureToggles.promQLScope ? expr : this.templateSrv.replace(this.enhanceExprWithAdHocFilters(filters, expr), variables, this.interpolateQueryExpr);
    return {
      ...target,
      ...config.featureToggles.promQLScope ? { adhocFilters: this.generateScopeFilters(filters) } : {},
      expr: exprWithAdhoc,
      interval: this.templateSrv.replace(target.interval, variables),
      legendFormat: this.templateSrv.replace(target.legendFormat, variables)
    };
  }
  getVariables() {
    return this.templateSrv.getVariables().map((v) => `$${v.name}`);
  }
  interpolateString(string, scopedVars) {
    return this.templateSrv.replace(string, scopedVars, this.interpolateQueryExpr);
  }
  interpolateExploreMetrics(fromExploreMetrics) {
    return (value = [], variable) => {
      if (typeof value === "string" && fromExploreMetrics) {
        if (variable.name === "filters") {
          return wrapUtf8Filters(value);
        }
        if (variable.name === "groupby") {
          return utf8Support(value);
        }
      }
      return this.interpolateQueryExpr(value, variable);
    };
  }
  isUsingRelativeTimeRange(range) {
    if (typeof range.raw.from !== "string" || typeof range.raw.to !== "string") {
      return false;
    }
    return range.raw.from.includes("now") || range.raw.to.includes("now");
  }
  getDefaultQuery(app) {
    const defaults2 = {
      refId: "A",
      expr: "",
      range: true,
      instant: false
    };
    if (app === CoreApp.UnifiedAlerting) {
      return {
        ...defaults2,
        instant: true,
        range: false
      };
    }
    if (app === CoreApp.Explore) {
      return {
        ...defaults2,
        instant: true,
        range: true
      };
    }
    return defaults2;
  }
}
function extractRuleMappingFromGroups(groups) {
  return groups.reduce(
    (mapping, group) => group.rules.filter((rule) => rule.type === "recording").reduce((acc, rule) => {
      var _a;
      const existingRule = (_a = acc[rule.name]) != null ? _a : [];
      existingRule.push({
        query: rule.query,
        labels: rule.labels
      });
      acc[rule.name] = existingRule;
      return acc;
    }, mapping),
    {}
  );
}
const extractResourceMatcher = (queries, adhocFilters) => {
  const metricMatch = populateMatchParamsFromQueries(queries);
  const labelFilters = adhocFilters.map((f) => ({
    label: f.key,
    value: f.value,
    op: f.operator
  }));
  const labelsMatch = renderLabelsWithoutBrackets(labelFilters);
  if (metricMatch.length === 0 && labelsMatch.length === 0) {
    return void 0;
  }
  return `{${[...metricMatch, ...labelsMatch].join(",")}}`;
};

export { PrometheusDatasource, extractResourceMatcher, extractRuleMappingFromGroups };
//# sourceMappingURL=datasource.mjs.map
