import { jsxs, jsx, Fragment } from 'react/jsx-runtime';
import { useState } from 'react';
import { selectors } from '@grafana/e2e-selectors';
import { t, Trans } from '@grafana/i18n';
import { DataSourcePicker, config } from '@grafana/runtime';
import { useTheme2, InlineField, Switch, Input, Button } from '@grafana/ui';
import { PROM_CONFIG_LABEL_WIDTH } from '../constants.mjs';
import { overhaulStyles, docsTip } from './shared/utils.mjs';

function ExemplarSetting({ value, onChange, onDelete, disabled }) {
  const [isInternalLink, setIsInternalLink] = useState(Boolean(value.datasourceUid));
  const theme = useTheme2();
  const styles = overhaulStyles(theme);
  return /* @__PURE__ */ jsxs("div", { className: "gf-form-group", children: [
    /* @__PURE__ */ jsx(
      InlineField,
      {
        label: t("grafana-prometheus.configuration.exemplar-setting.label-internal-link", "Internal link"),
        labelWidth: PROM_CONFIG_LABEL_WIDTH,
        disabled,
        tooltip: /* @__PURE__ */ jsxs(Fragment, { children: [
          /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.configuration.exemplar-setting.tooltip-internal-link", children: "Enable this option if you have an internal link. When enabled, this reveals the data source selector. Select the backend tracing data store for your exemplar data." }),
          " ",
          docsTip()
        ] }),
        interactive: true,
        className: styles.switchField,
        children: /* @__PURE__ */ jsx(Fragment, { children: /* @__PURE__ */ jsx(
          Switch,
          {
            value: isInternalLink,
            "data-testid": selectors.components.DataSource.Prometheus.configPage.internalLinkSwitch,
            onChange: (ev) => setIsInternalLink(ev.currentTarget.checked)
          }
        ) })
      }
    ),
    isInternalLink ? /* @__PURE__ */ jsx(
      InlineField,
      {
        label: t("grafana-prometheus.configuration.exemplar-setting.label-data-source", "Data source"),
        labelWidth: PROM_CONFIG_LABEL_WIDTH,
        tooltip: /* @__PURE__ */ jsxs(Fragment, { children: [
          /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.configuration.exemplar-setting.tooltip-data-source", children: "The data source the exemplar is going to navigate to." }),
          " ",
          docsTip()
        ] }),
        disabled,
        interactive: true,
        children: /* @__PURE__ */ jsx(
          DataSourcePicker,
          {
            filter: config.featureToggles.azureMonitorPrometheusExemplars ? void 0 : (ds) => ds.type !== "grafana-azure-monitor-datasource",
            tracing: true,
            current: value.datasourceUid,
            noDefault: true,
            width: 40,
            onChange: (ds) => onChange({
              ...value,
              datasourceUid: ds.uid,
              url: void 0
            })
          }
        )
      }
    ) : /* @__PURE__ */ jsx(
      InlineField,
      {
        label: t("grafana-prometheus.configuration.exemplar-setting.label-url", "URL"),
        labelWidth: PROM_CONFIG_LABEL_WIDTH,
        tooltip: /* @__PURE__ */ jsxs(Fragment, { children: [
          /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.configuration.exemplar-setting.tooltip-url", children: "The URL of the trace backend the user would go to see its trace" }),
          " ",
          docsTip()
        ] }),
        disabled,
        interactive: true,
        children: /* @__PURE__ */ jsx(
          Input,
          {
            placeholder: t(
              "grafana-prometheus.configuration.exemplar-setting.placeholder-httpsexamplecomvalueraw",
              "https://example.com/${__value.raw}"
            ),
            spellCheck: false,
            width: 40,
            value: value.url,
            onChange: (event) => onChange({
              ...value,
              datasourceUid: void 0,
              url: event.currentTarget.value
            })
          }
        )
      }
    ),
    /* @__PURE__ */ jsx(
      InlineField,
      {
        label: t("grafana-prometheus.configuration.exemplar-setting.label-url-label", "URL Label"),
        labelWidth: PROM_CONFIG_LABEL_WIDTH,
        tooltip: /* @__PURE__ */ jsxs(Fragment, { children: [
          /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.configuration.exemplar-setting.tooltip-url-label", children: "Use to override the button label on the exemplar traceID field." }),
          " ",
          docsTip()
        ] }),
        disabled,
        interactive: true,
        children: /* @__PURE__ */ jsx(
          Input,
          {
            placeholder: t(
              "grafana-prometheus.configuration.exemplar-setting.placeholder-go-to-examplecom",
              "Go to example.com"
            ),
            spellCheck: false,
            width: 40,
            value: value.urlDisplayLabel,
            onChange: (event) => onChange({
              ...value,
              urlDisplayLabel: event.currentTarget.value
            })
          }
        )
      }
    ),
    /* @__PURE__ */ jsx(
      InlineField,
      {
        label: t("grafana-prometheus.configuration.exemplar-setting.label-label-name", "Label name"),
        labelWidth: PROM_CONFIG_LABEL_WIDTH,
        tooltip: /* @__PURE__ */ jsxs(Fragment, { children: [
          /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.configuration.exemplar-setting.tooltip-label-name", children: "The name of the field in the labels object that should be used to get the traceID." }),
          " ",
          docsTip()
        ] }),
        disabled,
        interactive: true,
        children: /* @__PURE__ */ jsx(
          Input,
          {
            placeholder: t("grafana-prometheus.configuration.exemplar-setting.placeholder-trace-id", "traceID"),
            spellCheck: false,
            width: 40,
            value: value.name,
            onChange: (event) => onChange({
              ...value,
              name: event.currentTarget.value
            })
          }
        )
      }
    ),
    !disabled && /* @__PURE__ */ jsx(
      InlineField,
      {
        label: t(
          "grafana-prometheus.configuration.exemplar-setting.label-remove-exemplar-link",
          "Remove exemplar link"
        ),
        labelWidth: PROM_CONFIG_LABEL_WIDTH,
        disabled,
        children: /* @__PURE__ */ jsx(
          Button,
          {
            variant: "destructive",
            title: t(
              "grafana-prometheus.configuration.exemplar-setting.title-remove-exemplar-link",
              "Remove exemplar link"
            ),
            icon: "times",
            onClick: (event) => {
              event.preventDefault();
              onDelete();
            }
          }
        )
      }
    )
  ] });
}

export { ExemplarSetting };
//# sourceMappingURL=ExemplarSetting.mjs.map
