import { useRef, useState, useMemo, useEffect, useCallback } from 'react';
import { useDebounce } from 'react-use';
import { EMPTY_SELECTOR, LAST_USED_LABELS_KEY, METRIC_LABEL } from '../../constants.mjs';
import { buildSelector } from './selectorBuilder.mjs';

const useMetricsLabelsValues = (timeRange, languageProvider) => {
  const timeRangeRef = useRef(timeRange);
  const lastSeriesLimitRef = useRef(languageProvider.datasource.seriesLimit);
  const isInitializedRef = useRef(false);
  const [seriesLimit, setSeriesLimit] = useState(languageProvider.datasource.seriesLimit);
  const [err, setErr] = useState("");
  const [status, setStatus] = useState("Ready");
  const [validationStatus, setValidationStatus] = useState("");
  const [metrics, setMetrics] = useState([]);
  const [selectedMetric, setSelectedMetric] = useState("");
  const [labelKeys, setLabelKeys] = useState([]);
  const [selectedLabelKeys, setSelectedLabelKeys] = useState([]);
  const [lastSelectedLabelKey, setLastSelectedLabelKey] = useState("");
  const [labelValues, setLabelValues] = useState({});
  const [selectedLabelValues, setSelectedLabelValues] = useState({});
  const effectiveLimit = useMemo(() => seriesLimit, [seriesLimit]);
  useEffect(() => {
    if (timeRange.to.diff(timeRangeRef.current.to, "second") >= 5 && timeRange.from.diff(timeRangeRef.current.from, "second") >= 5) {
      timeRangeRef.current = timeRange;
    }
  }, [timeRange]);
  const handleError = useCallback((e, msg) => {
    if (e instanceof Error) {
      setErr(`${msg}: ${e.message}`);
    } else {
      setErr(`${msg}: Unknown error`);
    }
    setStatus("");
  }, []);
  const getMetricDetails = useCallback(
    (metricName) => {
      const meta = languageProvider.retrieveMetricsMetadata();
      return meta && meta[metricName] ? `(${meta[metricName].type}) ${meta[metricName].help}` : void 0;
    },
    [languageProvider]
  );
  const buildSafeSelector = useCallback((metric, labelValues2) => {
    const selector = buildSelector(metric, labelValues2);
    return selector === EMPTY_SELECTOR ? void 0 : selector;
  }, []);
  const loadSelectedLabelsFromStorage = useCallback(
    (availableLabelKeys) => {
      try {
        const labelKeysInLocalStorageAsString = localStorage.getItem(LAST_USED_LABELS_KEY) || "[]";
        const labelKeysInLocalStorage = JSON.parse(labelKeysInLocalStorageAsString);
        return labelKeysInLocalStorage.filter((slk) => availableLabelKeys.includes(slk));
      } catch (e) {
        handleError(e, "Failed to load saved label keys");
        return [];
      }
    },
    [handleError]
  );
  const fetchMetrics = useCallback(
    async (safeSelector) => {
      try {
        const fetchedMetrics = await languageProvider.queryLabelValues(
          timeRangeRef.current,
          METRIC_LABEL,
          safeSelector,
          effectiveLimit
        );
        return fetchedMetrics.map((m) => ({
          name: m,
          details: getMetricDetails(m)
        }));
      } catch (e) {
        handleError(e, "Error fetching metrics");
        return [];
      }
    },
    [getMetricDetails, handleError, languageProvider, effectiveLimit]
  );
  const fetchLabelKeys = useCallback(
    async (safeSelector) => {
      var _a;
      try {
        return (_a = await languageProvider.queryLabelKeys(timeRangeRef.current, safeSelector || void 0, effectiveLimit)) != null ? _a : [];
      } catch (e) {
        handleError(e, "Error fetching labels");
        return [];
      }
    },
    [handleError, languageProvider, effectiveLimit]
  );
  const fetchLabelValues = useCallback(
    async (labelKeys2, safeSelector) => {
      const transformedLabelValues = {};
      const newSelectedLabelValues = {};
      for (const lk of labelKeys2) {
        try {
          const values = await languageProvider.queryLabelValues(
            timeRangeRef.current,
            lk,
            safeSelector,
            effectiveLimit
          );
          transformedLabelValues[lk] = values;
          if (selectedLabelValues[lk]) {
            newSelectedLabelValues[lk] = [...selectedLabelValues[lk]];
          }
          setErr("");
        } catch (e) {
          handleError(e, "Error fetching label values");
        }
      }
      return [transformedLabelValues, newSelectedLabelValues];
    },
    [handleError, languageProvider, selectedLabelValues, effectiveLimit]
  );
  const initialize = useCallback(
    async (metric, labelValues2) => {
      const selector = buildSelector(metric, labelValues2);
      const safeSelector = selector === EMPTY_SELECTOR ? void 0 : selector;
      const transformedMetrics = await fetchMetrics(safeSelector);
      const transformedLabelKeys = await fetchLabelKeys(safeSelector);
      const labelKeysInLocalStorage = loadSelectedLabelsFromStorage(transformedLabelKeys);
      const [transformedLabelValues] = await fetchLabelValues(labelKeysInLocalStorage, safeSelector);
      setMetrics(transformedMetrics);
      setLabelKeys(transformedLabelKeys);
      setSelectedLabelKeys(labelKeysInLocalStorage);
      setLabelValues(transformedLabelValues);
    },
    [fetchLabelKeys, fetchLabelValues, fetchMetrics, loadSelectedLabelsFromStorage]
  );
  useEffect(() => {
    initialize(selectedMetric, selectedLabelValues);
    isInitializedRef.current = true;
  }, []);
  useDebounce(
    () => {
      if (isInitializedRef.current && lastSeriesLimitRef.current !== seriesLimit) {
        initialize(selectedMetric, selectedLabelValues);
        lastSeriesLimitRef.current = seriesLimit;
      }
    },
    300,
    [seriesLimit]
  );
  const handleSelectedMetricChange = async (metricName) => {
    const newSelectedMetric = selectedMetric !== metricName ? metricName : "";
    const selector = buildSafeSelector(newSelectedMetric, selectedLabelValues);
    try {
      const fetchedMetrics = await fetchMetrics(selector);
      const fetchedLabelKeys = await fetchLabelKeys(selector);
      const newSelectedLabelKeys = selectedLabelKeys.filter((slk) => fetchedLabelKeys.includes(slk));
      const [transformedLabelValues, newSelectedLabelValues] = await fetchLabelValues(
        newSelectedLabelKeys,
        newSelectedMetric === "" ? void 0 : selector
      );
      setMetrics(fetchedMetrics);
      setSelectedMetric(newSelectedMetric);
      setLabelKeys(fetchedLabelKeys);
      setSelectedLabelKeys(newSelectedLabelKeys);
      setLabelValues(transformedLabelValues);
      setSelectedLabelValues(newSelectedLabelValues);
    } catch (e) {
      handleError(e, "Error fetching labels");
    }
  };
  const handleSelectedLabelKeyChange = async (labelKey) => {
    const newSelectedLabelKeys = [...selectedLabelKeys];
    const lkIdx = newSelectedLabelKeys.indexOf(labelKey);
    const newLabelValues = { ...labelValues };
    const newSelectedLabelValues = { ...selectedLabelValues };
    if (lkIdx === -1) {
      newSelectedLabelKeys.push(labelKey);
      const safeSelector = buildSafeSelector(selectedMetric, selectedLabelValues);
      const [values] = await fetchLabelValues([labelKey], safeSelector);
      newLabelValues[labelKey] = values[labelKey];
    } else {
      newSelectedLabelKeys.splice(lkIdx, 1);
      delete newLabelValues[labelKey];
      delete newSelectedLabelValues[labelKey];
    }
    localStorage.setItem(LAST_USED_LABELS_KEY, JSON.stringify(newSelectedLabelKeys));
    setSelectedLabelKeys(newSelectedLabelKeys);
    setLabelValues(newLabelValues);
    setSelectedLabelValues(newSelectedLabelValues);
  };
  const handleSelectedLabelValueChange = async (labelKey, labelValue, isSelected) => {
    var _a;
    const newSelectedLabelValues = { ...selectedLabelValues };
    let newLastSelectedLabelKey = lastSelectedLabelKey;
    if (labelKey !== lastSelectedLabelKey) {
      newLastSelectedLabelKey = labelKey;
    }
    if (isSelected) {
      if (!newSelectedLabelValues[labelKey]) {
        newSelectedLabelValues[labelKey] = [];
      }
      newSelectedLabelValues[labelKey].push(labelValue);
    } else {
      newSelectedLabelValues[labelKey].splice(newSelectedLabelValues[labelKey].indexOf(labelValue), 1);
      if (newSelectedLabelValues[labelKey].length === 0) {
        delete newSelectedLabelValues[labelKey];
      }
    }
    let safeSelector = buildSafeSelector(selectedMetric, newSelectedLabelValues);
    let newLabelValues = {};
    if (selectedLabelKeys.length !== 0) {
      for (const lk of selectedLabelKeys) {
        try {
          const fetchedLabelValues = await languageProvider.queryLabelValues(
            timeRange,
            lk,
            safeSelector,
            effectiveLimit
          );
          if (newLastSelectedLabelKey === lk) {
            newLabelValues[lk] = Array.from(/* @__PURE__ */ new Set([...labelValues[lk], ...fetchedLabelValues]));
          } else {
            newLabelValues[lk] = fetchedLabelValues;
            newSelectedLabelValues[lk] = ((_a = newSelectedLabelValues[lk]) != null ? _a : []).filter(
              (item) => fetchedLabelValues.includes(item)
            );
          }
          setErr("");
        } catch (e) {
          handleError(e, "Error fetching label values");
        }
      }
    }
    safeSelector = buildSafeSelector(selectedMetric, newSelectedLabelValues);
    const newMetrics = await fetchMetrics(safeSelector);
    let newLabelKeys = [];
    if (!safeSelector) {
      newLabelKeys = await fetchLabelKeys(void 0);
    } else {
      const labelKeysSelector = `{${METRIC_LABEL}=~"${newMetrics.map((m) => m.name).join("|")}"}`;
      newLabelKeys = await fetchLabelKeys(labelKeysSelector);
    }
    const newSelectedLabelKeys = loadSelectedLabelsFromStorage(newLabelKeys);
    setMetrics(newMetrics);
    setLabelKeys(newLabelKeys);
    setSelectedLabelKeys(newSelectedLabelKeys);
    setLastSelectedLabelKey(newLastSelectedLabelKey);
    setLabelValues(newLabelValues);
    setSelectedLabelValues(newSelectedLabelValues);
  };
  const handleValidation = async () => {
    const selector = buildSelector(selectedMetric, selectedLabelValues);
    setValidationStatus(`Validating selector ${selector}`);
    setErr("");
    try {
      const results = await languageProvider.queryLabelKeys(timeRangeRef.current, selector, effectiveLimit);
      setValidationStatus(`Selector is valid (${Object.keys(results).length} labels found)`);
    } catch (e) {
      handleError(e, "Validation failed");
      setValidationStatus("");
    }
  };
  const handleClear = () => {
    localStorage.setItem(LAST_USED_LABELS_KEY, "[]");
    setSelectedMetric("");
    setSelectedLabelKeys([]);
    setSelectedLabelValues({});
    setErr("");
    setStatus("Ready");
    setValidationStatus("");
    initialize("", {});
  };
  return {
    err,
    setErr,
    status,
    setStatus,
    seriesLimit,
    setSeriesLimit,
    validationStatus,
    metrics,
    labelKeys,
    labelValues,
    selectedMetric,
    selectedLabelKeys,
    selectedLabelValues,
    handleSelectedMetricChange,
    handleSelectedLabelKeyChange,
    handleSelectedLabelValueChange,
    handleValidation,
    handleClear,
    // Helper functions - not part of the public API
    buildSafeSelector,
    loadSelectedLabelsFromStorage,
    fetchMetrics,
    fetchLabelKeys,
    fetchLabelValues
  };
};

export { useMetricsLabelsValues };
//# sourceMappingURL=useMetricsLabelsValues.mjs.map
