import { jsx, jsxs } from 'react/jsx-runtime';
import { useState, useMemo } from 'react';
import { FixedSizeList } from 'react-window';
import { selectors } from '@grafana/e2e-selectors';
import { t, Trans } from '@grafana/i18n';
import { useStyles2, Label, Input, BrowserLabel } from '@grafana/ui';
import { LIST_ITEM_SIZE } from '../../constants.mjs';
import { useMetricsBrowser } from './MetricsBrowserContext.mjs';
import { getStylesMetricSelector } from './styles.mjs';

function MetricSelector() {
  const styles = useStyles2(getStylesMetricSelector);
  const [metricSearchTerm, setMetricSearchTerm] = useState("");
  const { metrics, selectedMetric, seriesLimit, setSeriesLimit, onMetricClick } = useMetricsBrowser();
  const filteredMetrics = useMemo(() => {
    return metrics.filter((m) => m.name === selectedMetric || m.name.includes(metricSearchTerm));
  }, [metrics, selectedMetric, metricSearchTerm]);
  return /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsxs("div", { className: styles.section, children: [
    /* @__PURE__ */ jsx(
      Label,
      {
        description: t(
          "grafana-prometheus.components.metric-selector.label-select-metric",
          "Once a metric is selected only possible labels are shown. Labels are limited by the series limit below."
        ),
        children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.components.metric-selector.select-a-metric", children: "1. Select a metric" })
      }
    ),
    /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx(
      Input,
      {
        onChange: (e) => setMetricSearchTerm(e.currentTarget.value),
        "aria-label": t(
          "grafana-prometheus.components.metric-selector.aria-label-filter-expression-for-metric",
          "Filter expression for metric"
        ),
        value: metricSearchTerm,
        "data-testid": selectors.components.DataSource.Prometheus.queryEditor.code.metricsBrowser.selectMetric
      }
    ) }),
    /* @__PURE__ */ jsx(
      Label,
      {
        description: t(
          "grafana-prometheus.components.metric-selector.description-series-limit",
          "The limit applies to all metrics, labels, and values. Leave the field empty to use the default limit. Set to 0 to disable the limit and fetch everything \u2014 this may cause performance issues."
        ),
        children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.components.metric-selector.series-limit", children: "Series limit" })
      }
    ),
    /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx(
      Input,
      {
        onChange: (e) => setSeriesLimit(parseInt(e.currentTarget.value.trim(), 10)),
        "aria-label": t(
          "grafana-prometheus.components.metric-selector.aria-label-limit-results-from-series-endpoint",
          "Limit results from series endpoint"
        ),
        value: seriesLimit,
        "data-testid": selectors.components.DataSource.Prometheus.queryEditor.code.metricsBrowser.seriesLimit
      }
    ) }),
    /* @__PURE__ */ jsx(
      "div",
      {
        role: "list",
        className: styles.valueListWrapper,
        "data-testid": selectors.components.DataSource.Prometheus.queryEditor.code.metricsBrowser.metricList,
        children: /* @__PURE__ */ jsx(
          FixedSizeList,
          {
            height: Math.min(450, filteredMetrics.length * LIST_ITEM_SIZE),
            itemCount: filteredMetrics.length,
            itemSize: LIST_ITEM_SIZE,
            itemKey: (i) => filteredMetrics[i].name,
            width: 300,
            className: styles.valueList,
            children: ({ index, style }) => {
              const metric = filteredMetrics[index];
              return /* @__PURE__ */ jsx("div", { style, children: /* @__PURE__ */ jsx(
                BrowserLabel,
                {
                  name: metric.name,
                  value: metric.name,
                  title: metric.details,
                  active: metric.name === selectedMetric,
                  onClick: (name, value) => {
                    setMetricSearchTerm("");
                    onMetricClick(name);
                  },
                  searchTerm: metricSearchTerm
                }
              ) });
            }
          }
        )
      }
    )
  ] }) });
}

export { MetricSelector };
//# sourceMappingURL=MetricSelector.mjs.map
