import { sortDataFrame, getTimeField } from '../../dataframe/processDataFrame.mjs';
import { TIME_SERIES_VALUE_FIELD_NAME, FieldType } from '../../types/dataFrame.mjs';
import { fieldMatchers } from '../matchers.mjs';
import { FieldMatcherID } from '../matchers/ids.mjs';
import { JoinMode } from './joinByField.mjs';

function pickBestJoinField(data) {
  const { timeField } = getTimeField(data[0]);
  if (timeField) {
    return fieldMatchers.get(FieldMatcherID.firstTimeField).get({});
  }
  let common = [];
  for (const f of data[0].fields) {
    if (f.type === FieldType.number) {
      common.push(f.name);
    }
  }
  for (let i = 1; i < data.length; i++) {
    const names = [];
    for (const f of data[0].fields) {
      if (f.type === FieldType.number) {
        names.push(f.name);
      }
    }
    common = common.filter((v) => !names.includes(v));
  }
  return fieldMatchers.get(FieldMatcherID.byName).get(common[0]);
}
function getJoinMatcher(options) {
  var _a;
  return (_a = options.joinBy) != null ? _a : pickBestJoinField(options.frames);
}
function maybeSortFrame(frame, fieldIdx) {
  if (fieldIdx >= 0) {
    let sortByField = frame.fields[fieldIdx];
    if (sortByField.type !== FieldType.string && !isLikelyAscendingVector(sortByField.values)) {
      frame = sortDataFrame(frame, fieldIdx);
    }
  }
  return frame;
}
function joinDataFrames(options) {
  var _a, _b, _c, _d, _e;
  if (!((_a = options.frames) == null ? void 0 : _a.length)) {
    return;
  }
  const nullMode = (_b = options.nullMode) != null ? _b : (field) => {
    var _a2;
    let spanNulls = (_a2 = field.config.custom) == null ? void 0 : _a2.spanNulls;
    return spanNulls === true ? NULL_REMOVE : spanNulls === -1 ? NULL_RETAIN : NULL_EXPAND;
  };
  if (options.frames.length === 1) {
    let frame = options.frames[0];
    let frameCopy = frame;
    const joinFieldMatcher2 = getJoinMatcher(options);
    let joinIndex = frameCopy.fields.findIndex((f) => joinFieldMatcher2(f, frameCopy, options.frames));
    if (options.keepOriginIndices) {
      frameCopy = {
        ...frame,
        fields: frame.fields.map((f, fieldIndex) => {
          const copy = { ...f };
          const origin = {
            frameIndex: 0,
            fieldIndex
          };
          if (copy.state) {
            copy.state.origin = origin;
          } else {
            copy.state = { origin };
          }
          return copy;
        })
      };
      if (joinIndex > 0) {
        const joinField = frameCopy.fields[joinIndex];
        const fields = frameCopy.fields.filter((f, idx) => idx !== joinIndex);
        fields.unshift(joinField);
        frameCopy.fields = fields;
        joinIndex = 0;
      }
    }
    if (joinIndex >= 0) {
      frameCopy = maybeSortFrame(frameCopy, joinIndex);
    }
    if (options.keep) {
      let fields = frameCopy.fields.filter(
        (f, fieldIdx) => fieldIdx === joinIndex || options.keep(f, frameCopy, options.frames)
      );
      if (frame !== frameCopy) {
        frameCopy.fields = fields;
      } else {
        frameCopy = {
          ...frame,
          fields
        };
      }
    }
    return frameCopy;
  }
  const nullModes = [];
  const allData = [];
  const originalFields = [];
  const joinFieldMatcher = getJoinMatcher(options);
  for (let frameIndex = 0; frameIndex < options.frames.length; frameIndex++) {
    const frame = options.frames[frameIndex];
    if (!frame || !((_c = frame.fields) == null ? void 0 : _c.length)) {
      continue;
    }
    const nullModesFrame = [NULL_REMOVE];
    let join2 = void 0;
    let fields = [];
    for (let fieldIndex = 0; fieldIndex < frame.fields.length; fieldIndex++) {
      const field = frame.fields[fieldIndex];
      field.state = field.state || {};
      if (!join2 && joinFieldMatcher(field, frame, options.frames)) {
        join2 = field;
      } else {
        if (options.keep && !options.keep(field, frame, options.frames)) {
          continue;
        }
        nullModesFrame.push(nullMode(field));
        let labels = (_d = field.labels) != null ? _d : {};
        let name = field.name;
        if (frame.name) {
          if (field.name === TIME_SERIES_VALUE_FIELD_NAME) {
            name = frame.name;
          } else if (labels.name == null) {
            labels = { ...labels, name: frame.name };
          }
        }
        fields.push({
          ...field,
          name,
          labels
        });
      }
      if (options.keepOriginIndices) {
        field.state.origin = {
          frameIndex,
          fieldIndex
        };
      }
    }
    if (!join2) {
      continue;
    }
    if (originalFields.length === 0) {
      originalFields.push(join2);
    }
    nullModes.push(nullModesFrame);
    const a = [join2.values];
    for (const field of fields) {
      a.push(field.values);
      originalFields.push(field);
      if (!options.keepDisplayNames) {
        (_e = field.state) == null ? true : delete _e.displayName;
      }
    }
    allData.push(a);
  }
  let joined = [];
  if (options.mode === JoinMode.outerTabular) {
    joined = joinTabular(allData, true);
  } else if (options.mode === JoinMode.inner) {
    joined = joinTabular(allData);
  } else {
    joined = join(allData, nullModes, options.mode);
  }
  return {
    // ...options.data[0], // keep name, meta?
    length: joined[0] ? joined[0].length : 0,
    fields: originalFields.map((f, index) => ({
      ...f,
      values: joined[index]
    }))
  };
}
function joinTabular(tables, outer = false) {
  let ltable = tables[0];
  let lfield = ltable[0];
  for (let ti = 1; ti < tables.length; ti++) {
    let rtable = tables[ti];
    let rfield = rtable[0];
    let index = {};
    for (let i = 0; i < rfield.length; i++) {
      let val = rfield[i];
      let idxs = index[val];
      if (idxs == null) {
        idxs = index[val] = [];
      }
      idxs.push(i);
    }
    let matchedKeys = /* @__PURE__ */ new Set();
    let unmatchedLeft = [];
    let unmatchedRight = [];
    let matched = [];
    let count = 0;
    for (let i = 0; i < lfield.length; i++) {
      let v = lfield[i];
      if (v != null) {
        let idxs = index[v];
        if (idxs != null) {
          matched.push([i, idxs]);
          count += idxs.length;
          outer && matchedKeys.add(v);
        } else if (outer) {
          unmatchedLeft.push(i);
        }
      } else if (outer) {
        unmatchedLeft.push(i);
      }
    }
    count += unmatchedLeft.length;
    if (outer) {
      for (let k in index) {
        if (!matchedKeys.has(k)) {
          unmatchedRight.push(...index[k]);
        }
      }
      count += unmatchedRight.length;
    }
    let outFieldsTpl = Array.from({ length: ltable.length + rtable.length - 1 }, () => `Array(${count})`).join(",");
    let copyLeftRowTpl = ltable.map((c, i) => `joined[${i}][rowIdx] = ltable[${i}][lidx]`).join(";");
    let copyRightRowTpl = rtable.slice(1).map((c, i) => `joined[${ltable.length + i}][rowIdx] = rtable[${i + 1}][ridx]`).join(";");
    let nullLeftRowTpl = ltable.map((c, i) => `joined[${i}][rowIdx] = ${i === 0 ? `rtable[${i}][ridx]` : `null`}`).join(";");
    let nullRightRowTpl = rtable.slice(1).map((c, i) => `joined[${ltable.length + i}][rowIdx] = null`);
    let materialize = new Function(
      "matched",
      "unmatchedLeft",
      "unmatchedRight",
      "ltable",
      "rtable",
      `
      const joined = [${outFieldsTpl}];

      let rowIdx = 0;

      for (let i = 0; i < matched.length; i++) {
        let [lidx, ridxs] = matched[i];

        for (let j = 0; j < ridxs.length; j++, rowIdx++) {
          let ridx = ridxs[j];
          ${copyLeftRowTpl};
          ${copyRightRowTpl};
        }
      }

      for (let i = 0; i < unmatchedLeft.length; i++, rowIdx++) {
        let lidx = unmatchedLeft[i];
        ${copyLeftRowTpl};
        ${nullRightRowTpl};
      }

      for (let i = 0; i < unmatchedRight.length; i++, rowIdx++) {
        let ridx = unmatchedRight[i];
        ${nullLeftRowTpl};
        ${copyRightRowTpl};
      }

      return joined;
      `
    );
    let joined = materialize(matched, unmatchedLeft, unmatchedRight, ltable, rtable);
    ltable = joined;
    lfield = ltable[0];
  }
  return ltable;
}
const NULL_REMOVE = 0;
const NULL_RETAIN = 1;
const NULL_EXPAND = 2;
function nullExpand(yVals, nullIdxs, alignedLen) {
  for (let i = 0, xi, lastNullIdx = -1; i < nullIdxs.length; i++) {
    let nullIdx = nullIdxs[i];
    if (nullIdx > lastNullIdx) {
      xi = nullIdx - 1;
      while (xi >= 0 && yVals[xi] == null) {
        yVals[xi--] = null;
      }
      xi = nullIdx + 1;
      while (xi < alignedLen && yVals[xi] == null) {
        yVals[lastNullIdx = xi++] = null;
      }
    }
  }
}
function join(tables, nullModes, mode = JoinMode.outer) {
  let xVals = /* @__PURE__ */ new Set();
  for (let ti = 0; ti < tables.length; ti++) {
    let t = tables[ti];
    let xs = t[0];
    let len = xs.length;
    for (let i = 0; i < len; i++) {
      xVals.add(xs[i]);
    }
  }
  let data = [Array.from(xVals).sort((a, b) => a - b)];
  let alignedLen = data[0].length;
  let xIdxs = /* @__PURE__ */ new Map();
  for (let i = 0; i < alignedLen; i++) {
    xIdxs.set(data[0][i], i);
  }
  for (let ti = 0; ti < tables.length; ti++) {
    let t = tables[ti];
    let xs = t[0];
    for (let si = 1; si < t.length; si++) {
      let ys = t[si];
      let yVals = Array(alignedLen).fill(void 0);
      let nullMode = nullModes ? nullModes[ti][si] : NULL_RETAIN;
      let nullIdxs = [];
      for (let i = 0; i < ys.length; i++) {
        let yVal = ys[i];
        let alignedIdx = xIdxs.get(xs[i]);
        if (yVal === null) {
          if (nullMode !== NULL_REMOVE) {
            yVals[alignedIdx] = yVal;
            if (nullMode === NULL_EXPAND) {
              nullIdxs.push(alignedIdx);
            }
          }
        } else {
          yVals[alignedIdx] = yVal;
        }
      }
      nullExpand(yVals, nullIdxs, alignedLen);
      data.push(yVals);
    }
  }
  return data;
}
function isLikelyAscendingVector(data, samples = 50) {
  const len = data.length;
  if (len <= 1) {
    return true;
  }
  let firstIdx = 0;
  let lastIdx = len - 1;
  while (firstIdx <= lastIdx && data[firstIdx] == null) {
    firstIdx++;
  }
  while (lastIdx >= firstIdx && data[lastIdx] == null) {
    lastIdx--;
  }
  if (lastIdx <= firstIdx) {
    return true;
  }
  const stride = Math.max(1, Math.floor((lastIdx - firstIdx + 1) / samples));
  for (let prevVal = data[firstIdx], i = firstIdx + stride; i <= lastIdx; i += stride) {
    const v = data[i];
    if (v != null && prevVal != null) {
      if (v <= prevVal) {
        return false;
      }
      prevVal = v;
    }
  }
  return true;
}

export { NULL_EXPAND, NULL_REMOVE, NULL_RETAIN, isLikelyAscendingVector, join, joinDataFrames, maybeSortFrame, pickBestJoinField };
//# sourceMappingURL=joinDataFrames.mjs.map
