import uFuzzy from '@leeoniya/ufuzzy';
import { config } from '@grafana/runtime';
import { prometheusRegularEscape } from '../../../datasource.mjs';
import { escapeLabelValueInExactSelector } from '../../../language_utils.mjs';
import { FUNCTIONS } from '../../../promql.mjs';
import { isValidLegacyName } from '../../../utf8_support.mjs';
import { NeverCaseError } from './util.mjs';

const InsertAsSnippet = 4;
const metricNamesSearch = {
  // see https://github.com/leeoniya/uFuzzy?tab=readme-ov-file#how-it-works for details
  multiInsert: new uFuzzy({ intraMode: 0 }),
  singleError: new uFuzzy({ intraMode: 1 })
};
const snippetMarker = "${1:}";
function filterMetricNames({ metricNames, inputText, limit }) {
  if (!(inputText == null ? void 0 : inputText.trim())) {
    return metricNames.slice(0, limit);
  }
  const terms = metricNamesSearch.multiInsert.split(inputText);
  const isComplexSearch = terms.length > 4;
  const fuzzyResults = isComplexSearch ? metricNamesSearch.multiInsert.filter(metricNames, inputText) : metricNamesSearch.singleError.filter(metricNames, inputText);
  return fuzzyResults ? fuzzyResults.slice(0, limit).map((idx) => metricNames[idx]) : [];
}
function getAllMetricNamesCompletions(dataProvider) {
  let metricNames = dataProvider.getAllMetricNames();
  if (config.featureToggles.prometheusCodeModeMetricNamesSearch && metricNames.length > dataProvider.metricNamesSuggestionLimit) {
    const { monacoSettings } = dataProvider;
    monacoSettings.enableAutocompleteSuggestionsUpdate();
    if (monacoSettings.inputInRange) {
      metricNames = filterMetricNames({
        metricNames,
        inputText: monacoSettings.inputInRange,
        limit: dataProvider.metricNamesSuggestionLimit
      });
    } else {
      metricNames = metricNames.slice(0, dataProvider.metricNamesSuggestionLimit);
    }
  }
  return dataProvider.metricNamesToMetrics(metricNames).map((metric) => ({
    type: "METRIC_NAME",
    label: metric.name,
    detail: `${metric.name} : ${metric.type}`,
    documentation: metric.help,
    ...metric.isUtf8 ? {
      insertText: `{"${metric.name}"${snippetMarker}}`,
      insertTextRules: InsertAsSnippet
    } : {
      insertText: metric.name
    }
  }));
}
const FUNCTION_COMPLETIONS = FUNCTIONS.map((f) => {
  var _a;
  return {
    type: "FUNCTION",
    label: f.label,
    insertText: (_a = f.insertText) != null ? _a : "",
    // i don't know what to do when this is nullish. it should not be.
    detail: f.detail,
    documentation: f.documentation
  };
});
async function getAllFunctionsAndMetricNamesCompletions(dataProvider) {
  const metricNames = getAllMetricNamesCompletions(dataProvider);
  return [...FUNCTION_COMPLETIONS, ...metricNames];
}
const DURATION_COMPLETIONS = [
  "$__interval",
  "$__range",
  "$__rate_interval",
  "1m",
  "5m",
  "10m",
  "30m",
  "1h",
  "1d"
].map((text) => ({
  type: "DURATION",
  label: text,
  insertText: text
}));
function getAllHistoryCompletions(dataProvider) {
  const allHistory = dataProvider.getHistory();
  return allHistory.slice(0, 10).map((expr) => ({
    type: "HISTORY",
    label: expr,
    insertText: expr
  }));
}
function makeSelector(metricName, labels) {
  const allLabels = [...labels];
  if (metricName !== void 0) {
    allLabels.push({ name: "__name__", value: metricName, op: "=" });
  }
  const allLabelTexts = allLabels.map(
    (label) => `${label.name}${label.op}"${escapeLabelValueInExactSelector(label.value)}"`
  );
  return `{${allLabelTexts.join(",")}}`;
}
async function getLabelNames(metric, otherLabels, dataProvider, timeRange) {
  if (metric === void 0 && otherLabels.length === 0) {
    return Promise.resolve(dataProvider.getAllLabelNames());
  } else {
    const selector = makeSelector(metric, otherLabels);
    return await dataProvider.getSeriesLabels(timeRange, selector, otherLabels);
  }
}
async function getLabelNamesForCompletions(metric, suffix, triggerOnInsert, otherLabels, dataProvider, timeRange) {
  const labelNames = await getLabelNames(metric, otherLabels, dataProvider, timeRange);
  return labelNames.map((text) => {
    const isUtf8 = !isValidLegacyName(text);
    return {
      type: "LABEL_NAME",
      label: text,
      ...isUtf8 ? {
        insertText: `"${text}"${suffix}`,
        insertTextRules: InsertAsSnippet
      } : {
        insertText: `${text}${suffix}`
      },
      triggerOnInsert
    };
  });
}
async function getLabelNamesForSelectorCompletions(metric, otherLabels, dataProvider, timeRange) {
  return getLabelNamesForCompletions(metric, "=", true, otherLabels, dataProvider, timeRange);
}
async function getLabelNamesForByCompletions(metric, otherLabels, dataProvider, timeRange) {
  return getLabelNamesForCompletions(metric, "", false, otherLabels, dataProvider, timeRange);
}
async function getLabelValues(metric, labelName, otherLabels, dataProvider, timeRange) {
  if (metric === void 0 && otherLabels.length === 0) {
    return dataProvider.getLabelValues(timeRange, labelName);
  } else {
    const selector = makeSelector(metric, otherLabels);
    return await dataProvider.getSeriesValues(timeRange, labelName, selector);
  }
}
async function getLabelValuesForMetricCompletions(metric, labelName, betweenQuotes, otherLabels, dataProvider, timeRange) {
  const values = await getLabelValues(metric, labelName, otherLabels, dataProvider, timeRange);
  return values.map((text) => ({
    type: "LABEL_VALUE",
    label: text,
    insertText: formatLabelValueForCompletion(text, betweenQuotes)
  }));
}
function formatLabelValueForCompletion(value, betweenQuotes) {
  const text = config.featureToggles.prometheusSpecialCharsInLabelValues ? prometheusRegularEscape(value) : value;
  return betweenQuotes ? text : `"${text}"`;
}
function getCompletions(situation, dataProvider, timeRange) {
  switch (situation.type) {
    case "IN_DURATION":
      return Promise.resolve(DURATION_COMPLETIONS);
    case "IN_FUNCTION":
      return getAllFunctionsAndMetricNamesCompletions(dataProvider);
    case "AT_ROOT": {
      return getAllFunctionsAndMetricNamesCompletions(dataProvider);
    }
    case "EMPTY": {
      const metricNames = getAllMetricNamesCompletions(dataProvider);
      const historyCompletions = getAllHistoryCompletions(dataProvider);
      return Promise.resolve([...historyCompletions, ...FUNCTION_COMPLETIONS, ...metricNames]);
    }
    case "IN_LABEL_SELECTOR_NO_LABEL_NAME":
      return getLabelNamesForSelectorCompletions(situation.metricName, situation.otherLabels, dataProvider, timeRange);
    case "IN_GROUPING":
      return getLabelNamesForByCompletions(situation.metricName, situation.otherLabels, dataProvider, timeRange);
    case "IN_LABEL_SELECTOR_WITH_LABEL_NAME":
      return getLabelValuesForMetricCompletions(
        situation.metricName,
        situation.labelName,
        situation.betweenQuotes,
        situation.otherLabels,
        dataProvider,
        timeRange
      );
    default:
      throw new NeverCaseError(situation);
  }
}

export { filterMetricNames, getCompletions };
//# sourceMappingURL=completions.mjs.map
