import { jsx } from 'react/jsx-runtime';
import { cx, css } from '@emotion/css';
import { forwardRef, useCallback } from 'react';
import Highlighter from 'react-highlight-words';
import '@grafana/data';
import { useTheme2 } from '../../themes/ThemeContext.mjs';
import 'micro-memoize';
import '@emotion/react';
import 'tinycolor2';
import '../../utils/skeleton.mjs';
import { PartialHighlighter } from '../Typeahead/PartialHighlighter.mjs';

const Label = forwardRef(
  ({
    name,
    value,
    hidden,
    facets,
    onClick,
    className,
    loading,
    searchTerm,
    active,
    style,
    title,
    highlightParts,
    ...rest
  }, ref) => {
    const theme = useTheme2();
    const styles = getLabelStyles(theme);
    const searchWords = searchTerm ? [searchTerm] : [];
    const onLabelClick = useCallback(
      (event) => {
        if (onClick && !hidden) {
          onClick(name, value, event);
        }
      },
      [onClick, name, hidden, value]
    );
    let text = value || name;
    if (facets) {
      text = `${text} (${facets})`;
    }
    return /* @__PURE__ */ jsx(
      "button",
      {
        ref,
        onClick: onLabelClick,
        style,
        title: title || text,
        type: "button",
        role: "option",
        "aria-selected": !!active,
        className: cx(
          styles.base,
          active && styles.active,
          loading && styles.loading,
          hidden && styles.hidden,
          className,
          onClick && !hidden && styles.hover
        ),
        ...rest,
        children: highlightParts !== void 0 ? /* @__PURE__ */ jsx(PartialHighlighter, { text, highlightClassName: styles.matchHighLight, highlightParts }) : /* @__PURE__ */ jsx(
          Highlighter,
          {
            textToHighlight: text,
            searchWords,
            autoEscape: true,
            highlightClassName: styles.matchHighLight
          }
        )
      },
      text
    );
  }
);
Label.displayName = "Label";
const getLabelStyles = (theme) => ({
  base: css({
    display: "inline-block",
    cursor: "pointer",
    fontSize: theme.typography.size.sm,
    lineHeight: theme.typography.bodySmall.lineHeight,
    backgroundColor: theme.colors.background.secondary,
    color: theme.colors.text.primary,
    whiteSpace: "nowrap",
    textShadow: "none",
    padding: theme.spacing(0.5),
    borderRadius: theme.shape.radius.default,
    border: "none",
    marginRight: theme.spacing(1),
    marginBottom: theme.spacing(0.5)
  }),
  loading: css({
    fontWeight: theme.typography.fontWeightMedium,
    backgroundColor: theme.colors.primary.shade,
    color: theme.colors.text.primary,
    [theme.transitions.handleMotion("no-preference", "reduce")]: {
      animation: "pulse 3s ease-out 0s infinite normal forwards"
    },
    "@keyframes pulse": {
      "0%": {
        color: theme.colors.text.primary
      },
      "50%": {
        color: theme.colors.text.secondary
      },
      "100%": {
        color: theme.colors.text.disabled
      }
    }
  }),
  active: css({
    fontWeight: theme.typography.fontWeightMedium,
    backgroundColor: theme.colors.primary.main,
    color: theme.colors.primary.contrastText
  }),
  matchHighLight: css({
    background: "inherit",
    color: theme.components.textHighlight.text,
    backgroundColor: theme.components.textHighlight.background
  }),
  hidden: css({
    opacity: 0.6,
    cursor: "default",
    border: "1px solid transparent"
  }),
  hover: css({
    ["&:hover"]: {
      opacity: 0.85,
      cursor: "pointer"
    }
  })
});

export { Label };
//# sourceMappingURL=Label.mjs.map
