import { jsx } from 'react/jsx-runtime';
import { cx, css } from '@emotion/css';
import RcPicker from 'rc-picker';
import generateConfig from 'rc-picker/lib/generate/moment';
import locale from 'rc-picker/lib/locale/en_US';
import { dateTimeAsMoment, isDateTimeInput, dateTime } from '@grafana/data';
import { useStyles2 } from '../../themes/ThemeContext.mjs';
import { getFocusStyles } from '../../themes/mixins.mjs';
import { inputSizes } from '../Forms/commonStyles.mjs';
import { Icon } from '../Icon/Icon.mjs';
import 'rc-picker/assets/index.css';

const POPUP_CLASS_NAME = "time-of-day-picker-panel";
const TimeOfDayPicker = ({
  minuteStep = 1,
  showHour = true,
  showSeconds = false,
  value,
  size = "auto",
  disabled,
  disabledHours,
  disabledMinutes,
  disabledSeconds,
  placeholder,
  // note: we can't destructure allowEmpty/onChange here
  // in order to discriminate the types properly later in the onChange handler
  ...restProps
}) => {
  var _a;
  const styles = useStyles2(getStyles);
  const allowClear = (_a = restProps.allowEmpty) != null ? _a : false;
  return /* @__PURE__ */ jsx(
    RcPicker,
    {
      generateConfig,
      locale,
      allowClear: allowClear && {
        clearIcon: /* @__PURE__ */ jsx(Icon, { name: "times", className: styles.clearIcon })
      },
      className: cx(inputSizes()[size], styles.input),
      classNames: {
        popup: cx(styles.picker, POPUP_CLASS_NAME)
      },
      defaultValue: restProps.allowEmpty ? void 0 : dateTimeAsMoment(),
      disabled,
      disabledTime: () => ({
        disabledHours,
        disabledMinutes,
        disabledSeconds
      }),
      format: generateFormat(showHour, showSeconds),
      minuteStep,
      onChange: (value2) => {
        if (isDateTimeInput(value2)) {
          if (restProps.allowEmpty) {
            return restProps.onChange(value2 ? dateTime(value2) : void 0);
          } else {
            return restProps.onChange(dateTime(value2));
          }
        }
      },
      picker: "time",
      placeholder,
      showNow: false,
      needConfirm: false,
      suffixIcon: /* @__PURE__ */ jsx(Caret, { wrapperStyle: styles.caretWrapper }),
      value: value ? dateTimeAsMoment(value) : value
    }
  );
};
function generateFormat(showHour = true, showSeconds = false) {
  const maybeHour = showHour ? "HH:" : "";
  const maybeSecond = showSeconds ? ":ss" : "";
  return maybeHour + "mm" + maybeSecond;
}
const Caret = ({ wrapperStyle = "" }) => {
  return /* @__PURE__ */ jsx("div", { className: wrapperStyle, children: /* @__PURE__ */ jsx(Icon, { name: "angle-down" }) });
};
const getStyles = (theme) => {
  const bgColor = theme.components.input.background;
  const optionBgHover = theme.colors.action.hover;
  const borderRadius = theme.shape.radius.default;
  const borderColor = theme.components.input.borderColor;
  return {
    caretWrapper: css({
      position: "relative",
      top: "50%",
      transform: "translateY(-50%)",
      display: "inline-block",
      color: theme.colors.text.secondary
    }),
    clearIcon: css({
      color: theme.colors.text.secondary,
      "&:hover": {
        color: theme.colors.text.maxContrast
      }
    }),
    picker: css({
      "&.rc-picker-dropdown": {
        boxShadow: "none",
        zIndex: theme.zIndex.portal
      },
      ".rc-picker-time-panel-column": {
        fontSize: theme.typography.htmlFontSize,
        backgroundColor: bgColor,
        color: theme.colors.text.secondary,
        padding: "unset",
        overflowY: "auto",
        scrollbarWidth: "thin",
        width: theme.spacing(8),
        li: {
          paddingRight: theme.spacing(2),
          width: "auto",
          "&.rc-picker-time-panel-cell-selected": {
            backgroundColor: "inherit",
            border: `1px solid ${theme.colors.action.selectedBorder}`,
            borderRadius,
            color: theme.colors.text.primary
          },
          "&:hover": {
            background: optionBgHover,
            color: theme.colors.text.primary
          },
          "&.rc-picker-time-panel-cell-disabled": {
            color: theme.colors.action.disabledText
          }
        },
        ".rc-picker-time-panel-cell-inner": {
          color: "inherit"
        },
        "&:not(:last-of-type)": {
          borderRight: `1px solid ${borderColor}`
        }
      },
      ".rc-picker-panel": {
        boxShadow: theme.shadows.z3,
        backgroundColor: bgColor,
        borderColor,
        borderRadius,
        overflow: "hidden"
      }
    }),
    input: css({
      "&.rc-picker-focused": {
        border: "none",
        ".rc-picker-input": getFocusStyles(theme)
      },
      "&.rc-picker-disabled": {
        ".rc-picker-input": {
          backgroundColor: theme.colors.action.disabledBackground,
          color: theme.colors.action.disabledText,
          border: `1px solid ${theme.colors.action.disabledBackground}`,
          "&:focus": {
            boxShadow: "none"
          }
        }
      },
      ".rc-picker-input": {
        backgroundColor: bgColor,
        borderRadius,
        borderColor,
        borderStyle: "solid",
        borderWidth: "1px",
        color: theme.colors.text.primary,
        height: theme.spacing(4),
        padding: theme.spacing(0, 1),
        input: {
          color: "unset",
          backgroundColor: "unset",
          "&:focus": {
            outline: "none"
          },
          "&::placeholder": {
            color: theme.colors.text.disabled
          }
        }
      },
      ".rc-picker-clear": {
        alignItems: "center",
        display: "flex",
        insetInlineEnd: "unset",
        position: "relative"
      }
    })
  };
};

export { POPUP_CLASS_NAME, TimeOfDayPicker };
//# sourceMappingURL=TimeOfDayPicker.mjs.map
