import { map } from 'rxjs/operators';
import { getFieldDisplayName } from '../../field/fieldState.mjs';
import { FieldType } from '../../types/dataFrame.mjs';
import { DataTransformerID } from './ids.mjs';

const labelsToFieldsTransformer = {
  id: DataTransformerID.labelsToFields,
  name: "Labels to fields",
  description: "Extract time series labels to fields (columns or rows)",
  defaultOptions: {},
  operator: (options, ctx) => (source) => source.pipe(map((data) => labelsToFieldsTransformer.transformer(options, ctx)(data))),
  transformer: (options) => (data) => {
    var _a, _b;
    if (options.mode === "rows" /* Rows */) {
      return convertLabelsToRows(data, options.keepLabels);
    }
    const result = [];
    const keepLabels = ((_a = options.keepLabels) == null ? void 0 : _a.length) ? new Set(options.keepLabels) : void 0;
    for (const frame of data) {
      const newFields = [];
      const uniqueLabels = {};
      for (const field of frame.fields) {
        if (!field.labels) {
          newFields.push(field);
          continue;
        }
        const sansLabels = {
          ...field,
          config: {
            ...field.config,
            // we need to clear these for this transform as these can contain label names that we no longer want
            displayName: void 0,
            displayNameFromDS: void 0
          },
          labels: void 0
        };
        newFields.push(sansLabels);
        for (const labelName of Object.keys(field.labels)) {
          if (keepLabels && !keepLabels.has(labelName)) {
            continue;
          }
          if (options.valueLabel === labelName) {
            sansLabels.name = field.labels[labelName];
            continue;
          }
          const uniqueValues = (_b = uniqueLabels[labelName]) != null ? _b : uniqueLabels[labelName] = /* @__PURE__ */ new Set();
          uniqueValues.add(field.labels[labelName]);
        }
      }
      for (const name in uniqueLabels) {
        for (const value of uniqueLabels[name]) {
          const values = new Array(frame.length).fill(value);
          newFields.push({
            name,
            type: FieldType.string,
            values,
            config: {}
          });
        }
      }
      result.push({
        ...frame,
        fields: newFields,
        length: frame.length
      });
    }
    return result;
  }
};
function convertLabelsToRows(data, keepLabels) {
  const result = [];
  for (const frame of data) {
    for (const field of frame.fields) {
      if (field.labels) {
        const keys = [];
        const vals = [];
        if (keepLabels) {
          for (const key of keepLabels) {
            keys.push(key);
            vals.push(field.labels[key]);
          }
        } else {
          for (const [key, val] of Object.entries(field.labels)) {
            keys.push(key);
            vals.push(val);
          }
        }
        if (vals.length) {
          result.push({
            ...frame,
            name: getFieldDisplayName(field, frame, data),
            fields: [
              { name: "label", type: FieldType.string, config: {}, values: keys },
              { name: "value", type: FieldType.string, config: {}, values: vals }
            ],
            length: vals.length
          });
        }
      }
    }
  }
  return result;
}

export { labelsToFieldsTransformer };
//# sourceMappingURL=labelsToFields.mjs.map
