import { isValidDuration, parseDuration, durationToMilliseconds, dateTime, incrRoundDn, amendTable, trimTable, rangeUtil } from '@grafana/data';

const defaultPrometheusQueryOverlapWindow = "10m";
const getFieldIdentity = (field) => {
  var _a;
  return `${field.type}|${field.name}|${JSON.stringify((_a = field.labels) != null ? _a : "")}`;
};
class QueryCache {
  constructor(options) {
    this.applyInterpolation = (str, scopedVars) => str;
    this.cache = /* @__PURE__ */ new Map();
    const unverifiedOverlap = options.overlapString;
    if (isValidDuration(unverifiedOverlap)) {
      const duration = parseDuration(unverifiedOverlap);
      this.overlapWindowMs = durationToMilliseconds(duration);
    } else {
      const duration = parseDuration(defaultPrometheusQueryOverlapWindow);
      this.overlapWindowMs = durationToMilliseconds(duration);
    }
    this.getTargetSignature = options.getTargetSignature;
    if (options.applyInterpolation) {
      this.applyInterpolation = options.applyInterpolation;
    }
  }
  // can be used to change full range request to partial, split into multiple requests
  requestInfo(request) {
    var _a, _b, _c;
    const newFrom = request.range.from.valueOf();
    const newTo = request.range.to.valueOf();
    const shouldCache = ((_b = (_a = request.rangeRaw) == null ? void 0 : _a.to) == null ? void 0 : _b.toString()) === "now";
    let doPartialQuery = shouldCache;
    let prevTo = void 0;
    const reqTargetSignatures = /* @__PURE__ */ new Map();
    request.targets.forEach((target) => {
      let targetIdentity = `${request.dashboardUID}|${request.panelId}|${target.refId}`;
      let targetSignature = this.getTargetSignature(request, target);
      reqTargetSignatures.set(targetIdentity, targetSignature);
    });
    for (const [targetIdentity, targetSignature] of reqTargetSignatures) {
      let cached = this.cache.get(targetIdentity);
      let cachedSig = cached == null ? void 0 : cached.signature;
      if (cachedSig !== targetSignature) {
        doPartialQuery = false;
      } else {
        prevTo = (_c = cached == null ? void 0 : cached.prevTo) != null ? _c : Infinity;
        doPartialQuery = newTo > prevTo && newFrom <= prevTo;
      }
      if (!doPartialQuery) {
        break;
      }
    }
    if (doPartialQuery && prevTo) {
      let newFromPartial = Math.max(prevTo - this.overlapWindowMs, newFrom);
      const newToDate = dateTime(newTo);
      const newFromPartialDate = dateTime(incrRoundDn(newFromPartial, request.intervalMs));
      request = {
        ...request,
        range: {
          ...request.range,
          from: newFromPartialDate,
          to: newToDate
        }
      };
    } else {
      reqTargetSignatures.forEach((targSig, targIdent) => {
        this.cache.delete(targIdent);
      });
    }
    return {
      requests: [request],
      targetSignatures: reqTargetSignatures,
      shouldCache
    };
  }
  // should amend existing cache with new frames and return full response
  procFrames(request, requestInfo, respFrames) {
    if (requestInfo == null ? void 0 : requestInfo.shouldCache) {
      const newFrom = request.range.from.valueOf();
      const newTo = request.range.to.valueOf();
      const respByTarget = /* @__PURE__ */ new Map();
      respFrames.forEach((frame) => {
        let targetIdent = `${request.dashboardUID}|${request.panelId}|${frame.refId}`;
        let frames = respByTarget.get(targetIdent);
        if (!frames) {
          frames = [];
          respByTarget.set(targetIdent, frames);
        }
        frames.push(frame);
      });
      let outFrames = [];
      respByTarget.forEach((respFrames2, targetIdentity) => {
        var _a, _b;
        let cachedFrames = (_b = targetIdentity ? (_a = this.cache.get(targetIdentity)) == null ? void 0 : _a.frames : null) != null ? _b : [];
        respFrames2.forEach((respFrame) => {
          if (respFrame.length === 0 || respFrame.fields.length === 0) {
            return;
          }
          let respFrameIdentity = getFieldIdentity(respFrame.fields[1]);
          let cachedFrame = cachedFrames.find((cached) => getFieldIdentity(cached.fields[1]) === respFrameIdentity);
          if (!cachedFrame) {
            cachedFrames.push(respFrame);
          } else {
            let prevTable = cachedFrame.fields.map((field) => field.values);
            let nextTable = respFrame.fields.map((field) => field.values);
            let amendedTable = amendTable(prevTable, nextTable);
            if (amendedTable) {
              for (let i = 0; i < amendedTable.length; i++) {
                cachedFrame.fields[i].values = amendedTable[i];
                if (cachedFrame.fields[i].config.displayNameFromDS !== respFrame.fields[i].config.displayNameFromDS) {
                  cachedFrame.fields[i].config.displayNameFromDS = respFrame.fields[i].config.displayNameFromDS;
                }
              }
              cachedFrame.length = cachedFrame.fields[0].values.length;
            }
          }
        });
        let nonEmptyCachedFrames = [];
        cachedFrames.forEach((frame) => {
          let table = frame.fields.map((field) => field.values);
          const dataPointStep = findDatapointStep(request, respFrames2, this.applyInterpolation);
          let trimmed = trimTable(table, newFrom - dataPointStep, newTo);
          if (trimmed[0].length > 0) {
            for (let i = 0; i < trimmed.length; i++) {
              frame.fields[i].values = trimmed[i];
            }
            nonEmptyCachedFrames.push(frame);
          }
        });
        this.cache.set(targetIdentity, {
          signature: requestInfo.targetSignatures.get(targetIdentity),
          frames: nonEmptyCachedFrames,
          prevTo: newTo
        });
        outFrames.push(...nonEmptyCachedFrames);
      });
      respFrames = outFrames.map((frame) => ({
        ...frame,
        fields: frame.fields.map((field) => ({
          ...field,
          config: {
            ...field.config
            // prevents mutatative exemplars links (re)enrichment
          },
          values: field.values.slice()
        }))
      }));
    }
    return respFrames;
  }
}
function findDatapointStep(request, respFrames, applyInterpolation) {
  var _a;
  if (((_a = request.targets[0].datasource) == null ? void 0 : _a.type) !== "prometheus") {
    return 0;
  }
  const target = request.targets.find((t) => t.refId === respFrames[0].refId);
  let dataPointStep = request.intervalMs;
  if (target == null ? void 0 : target.interval) {
    const minStepMs = rangeUtil.intervalToMs(applyInterpolation(target.interval));
    if (minStepMs > request.intervalMs) {
      dataPointStep = minStepMs;
    }
  }
  return dataPointStep;
}

export { QueryCache, defaultPrometheusQueryOverlapWindow, findDatapointStep, getFieldIdentity };
//# sourceMappingURL=QueryCache.js.map
