import { screen, waitFor } from '@testing-library/react';

import { config, locationService } from '@grafana/runtime';

import { ScopesApiClient } from '../ScopesApiClient';
import { ScopesService } from '../ScopesService';
import { ScopesDashboardsService } from '../dashboards/ScopesDashboardsService';
import { ScopeNavigation } from '../dashboards/types';

import {
  clearNotFound,
  expandDashboardFolder,
  searchDashboards,
  toggleDashboards,
  updateScopes,
} from './utils/actions';
import {
  expectDashboardFolderNotInDocument,
  expectDashboardInDocument,
  expectDashboardLength,
  expectDashboardNotInDocument,
  expectDashboardsClosed,
  expectDashboardSearchValue,
  expectDashboardsOpen,
  expectDashboardsSearch,
  expectNoDashboardsForFilter,
  expectNoDashboardsForScope,
  expectNoDashboardsNoScopes,
} from './utils/assertions';
import {
  alternativeDashboardWithRootFolder,
  alternativeDashboardWithTwoFolders,
  dashboardWithOneFolder,
  dashboardWithoutFolder,
  dashboardWithRootFolder,
  dashboardWithRootFolderAndOtherFolder,
  dashboardWithTwoFolders,
  getDatasource,
  getInstanceSettings,
  getMock,
  navigationWithSubScope,
  navigationWithSubScope2,
  navigationWithSubScopeDifferent,
  navigationWithSubScopeAndGroups,
  subScopeMimirItems,
} from './utils/mocks';
import { renderDashboard, resetScenes } from './utils/render';

jest.mock('@grafana/runtime', () => ({
  __esModule: true,
  ...jest.requireActual('@grafana/runtime'),
  useChromeHeaderHeight: jest.fn(),
  getBackendSrv: () => ({ get: getMock }),
  getDataSourceSrv: () => ({ get: getDatasource, getInstanceSettings }),
  usePluginLinks: jest.fn().mockReturnValue({ links: [] }),
}));

describe('Dashboards list', () => {
  let fetchDashboardsSpy: jest.SpyInstance;
  let fetchScopeNavigationsSpy: jest.SpyInstance;
  let scopesService: ScopesService;
  let scopesDashboardsService: ScopesDashboardsService;
  let apiClient: ScopesApiClient;

  beforeAll(() => {
    config.featureToggles.scopeFilters = true;
    config.featureToggles.groupByVariable = true;
  });

  beforeEach(async () => {
    const result = await renderDashboard();
    scopesService = result.scopesService;
    scopesDashboardsService = result.scopesDashboardsService;
    apiClient = result.client;
    fetchDashboardsSpy = jest.spyOn(apiClient, 'fetchDashboards');
    fetchScopeNavigationsSpy = jest.spyOn(apiClient, 'fetchScopeNavigations');
  });

  afterEach(async () => {
    locationService.replace('');
    await resetScenes([fetchDashboardsSpy, fetchScopeNavigationsSpy]);
  });

  it('Opens container and fetches dashboards list when a scope is selected', async () => {
    expectDashboardsClosed();
    await updateScopes(scopesService, ['mimir']);
    expectDashboardsOpen();
    expect(fetchDashboardsSpy).toHaveBeenCalled();
  });

  it('Closes container when no scopes are selected', async () => {
    await updateScopes(scopesService, ['mimir']);
    expectDashboardsOpen();
    await updateScopes(scopesService, ['mimir', 'loki']);
    expectDashboardsOpen();
    await updateScopes(scopesService, []);
    expectDashboardsClosed();
  });

  it('Fetches dashboards list when the list is expanded', async () => {
    await toggleDashboards();
    await updateScopes(scopesService, ['mimir']);
    expect(fetchDashboardsSpy).toHaveBeenCalled();
  });

  it('Fetches dashboards list when the list is expanded after scope selection', async () => {
    await updateScopes(scopesService, ['mimir']);
    await toggleDashboards();
    expect(fetchDashboardsSpy).toHaveBeenCalled();
  });

  it('Shows dashboards for multiple scopes', async () => {
    await toggleDashboards();
    await updateScopes(scopesService, ['grafana']);
    await expandDashboardFolder('General');
    await expandDashboardFolder('Observability');
    await expandDashboardFolder('Usage');
    expectDashboardFolderNotInDocument('Components');
    expectDashboardFolderNotInDocument('Investigations');
    expectDashboardInDocument('general-data-sources');
    expectDashboardInDocument('general-usage');
    expectDashboardInDocument('observability-backend-errors');
    expectDashboardInDocument('observability-backend-logs');
    expectDashboardInDocument('observability-frontend-errors');
    expectDashboardInDocument('observability-frontend-logs');
    expectDashboardInDocument('usage-data-sources');
    expectDashboardInDocument('usage-stats');
    expectDashboardInDocument('usage-usage-overview');
    expectDashboardInDocument('frontend');
    expectDashboardInDocument('overview');
    expectDashboardInDocument('stats');
    expectDashboardNotInDocument('multiple3-datasource-errors');
    expectDashboardNotInDocument('multiple4-datasource-logs');
    expectDashboardNotInDocument('multiple0-ingester');
    expectDashboardNotInDocument('multiple1-distributor');
    expectDashboardNotInDocument('multiple2-compacter');
    expectDashboardNotInDocument('another-stats');

    await updateScopes(scopesService, ['grafana', 'mimir']);
    await expandDashboardFolder('General');
    await expandDashboardFolder('Observability');
    await expandDashboardFolder('Usage');
    await expandDashboardFolder('Components');
    await expandDashboardFolder('Investigations');
    expectDashboardInDocument('general-data-sources');
    expectDashboardInDocument('general-usage');
    expectDashboardInDocument('observability-backend-errors');
    expectDashboardInDocument('observability-backend-logs');
    expectDashboardInDocument('observability-frontend-errors');
    expectDashboardInDocument('observability-frontend-logs');
    expectDashboardInDocument('usage-data-sources');
    expectDashboardInDocument('usage-stats');
    expectDashboardInDocument('usage-usage-overview');
    expectDashboardInDocument('frontend');
    expectDashboardInDocument('overview');
    expectDashboardInDocument('stats');
    expectDashboardLength('multiple3-datasource-errors', 2);
    expectDashboardLength('multiple4-datasource-logs', 2);
    expectDashboardLength('multiple0-ingester', 2);
    expectDashboardLength('multiple1-distributor', 2);
    expectDashboardLength('multiple2-compacter', 2);
    expectDashboardInDocument('another-stats');

    await updateScopes(scopesService, ['grafana']);
    await expandDashboardFolder('General');
    await expandDashboardFolder('Observability');
    await expandDashboardFolder('Usage');
    expectDashboardFolderNotInDocument('Components');
    expectDashboardFolderNotInDocument('Investigations');
    expectDashboardInDocument('general-data-sources');
    expectDashboardInDocument('general-usage');
    expectDashboardInDocument('observability-backend-errors');
    expectDashboardInDocument('observability-backend-logs');
    expectDashboardInDocument('observability-frontend-errors');
    expectDashboardInDocument('observability-frontend-logs');
    expectDashboardInDocument('usage-data-sources');
    expectDashboardInDocument('usage-stats');
    expectDashboardInDocument('usage-usage-overview');
    expectDashboardInDocument('frontend');
    expectDashboardInDocument('overview');
    expectDashboardInDocument('stats');
    expectDashboardFolderNotInDocument('multiple3-datasource-errors');
    expectDashboardFolderNotInDocument('multiple4-datasource-logs');
    expectDashboardFolderNotInDocument('multiple0-ingester');
    expectDashboardFolderNotInDocument('multiple1-distributor');
    expectDashboardFolderNotInDocument('multiple2-compacter');
    expectDashboardFolderNotInDocument('another-stats');
  });

  it('Filters the dashboards list for dashboards', async () => {
    await toggleDashboards();
    await updateScopes(scopesService, ['grafana']);
    await expandDashboardFolder('General');
    await expandDashboardFolder('Observability');
    await expandDashboardFolder('Usage');
    expectDashboardInDocument('general-data-sources');
    expectDashboardInDocument('general-usage');
    expectDashboardInDocument('observability-backend-errors');
    expectDashboardInDocument('observability-backend-logs');
    expectDashboardInDocument('observability-frontend-errors');
    expectDashboardInDocument('observability-frontend-logs');
    expectDashboardInDocument('usage-data-sources');
    expectDashboardInDocument('usage-stats');
    expectDashboardInDocument('usage-usage-overview');
    expectDashboardInDocument('frontend');
    expectDashboardInDocument('overview');
    expectDashboardInDocument('stats');

    await searchDashboards('Stats');
    expectDashboardFolderNotInDocument('general-data-sources');
    expectDashboardFolderNotInDocument('general-usage');
    expectDashboardFolderNotInDocument('observability-backend-errors');
    expectDashboardFolderNotInDocument('observability-backend-logs');
    expectDashboardFolderNotInDocument('observability-frontend-errors');
    expectDashboardFolderNotInDocument('observability-frontend-logs');
    expectDashboardFolderNotInDocument('usage-data-sources');
    expectDashboardInDocument('usage-stats');
    expectDashboardFolderNotInDocument('usage-usage-overview');
    expectDashboardFolderNotInDocument('frontend');
    expectDashboardFolderNotInDocument('overview');
    expectDashboardInDocument('stats');
  });

  it('Filters the dashboards list for folders', async () => {
    await toggleDashboards();
    await updateScopes(scopesService, ['grafana']);
    await expandDashboardFolder('General');
    await expandDashboardFolder('Observability');
    await expandDashboardFolder('Usage');
    expectDashboardInDocument('general-data-sources');
    expectDashboardInDocument('general-usage');
    expectDashboardInDocument('observability-backend-errors');
    expectDashboardInDocument('observability-backend-logs');
    expectDashboardInDocument('observability-frontend-errors');
    expectDashboardInDocument('observability-frontend-logs');
    expectDashboardInDocument('usage-data-sources');
    expectDashboardInDocument('usage-stats');
    expectDashboardInDocument('usage-usage-overview');
    expectDashboardInDocument('frontend');
    expectDashboardInDocument('overview');
    expectDashboardInDocument('stats');

    await searchDashboards('Usage');
    expectDashboardFolderNotInDocument('general-data-sources');
    expectDashboardInDocument('general-usage');
    expectDashboardFolderNotInDocument('observability-backend-errors');
    expectDashboardFolderNotInDocument('observability-backend-logs');
    expectDashboardFolderNotInDocument('observability-frontend-errors');
    expectDashboardFolderNotInDocument('observability-frontend-logs');
    expectDashboardInDocument('usage-data-sources');
    expectDashboardInDocument('usage-stats');
    expectDashboardInDocument('usage-usage-overview');
    expectDashboardFolderNotInDocument('frontend');
    expectDashboardFolderNotInDocument('overview');
    expectDashboardFolderNotInDocument('stats');
  });

  it('Deduplicates the dashboards list', async () => {
    await toggleDashboards();
    await updateScopes(scopesService, ['dev', 'ops']);
    await expandDashboardFolder('Cardinality Management');
    await expandDashboardFolder('Usage Insights');
    expectDashboardLength('cardinality-management-labels', 1);
    expectDashboardLength('cardinality-management-metrics', 1);
    expectDashboardLength('cardinality-management-overview', 1);
    expectDashboardLength('usage-insights-alertmanager', 1);
    expectDashboardLength('usage-insights-data-sources', 1);
    expectDashboardLength('usage-insights-metrics-ingestion', 1);
    expectDashboardLength('usage-insights-overview', 1);
    expectDashboardLength('usage-insights-query-errors', 1);
    expectDashboardLength('billing-usage', 1);
  });

  it('does not redirect when scopes are set programmatically on dashboard not in scope navigation list', async () => {
    // Render another dashboard, which is not a scope navigation
    const mockNavigations: ScopeNavigation[] = [
      {
        spec: {
          scope: 'grafana',
          url: '/d/dashboard1',
        },
        status: {
          title: 'Dashboard 1',
          groups: ['group1'],
        },
        metadata: {
          name: 'dashboard1',
        },
      },
    ];
    fetchDashboardsSpy.mockResolvedValue(mockNavigations);

    await renderDashboard();
    expect(locationService.getLocation().pathname).toBe('/');

    // When scopes are set programmatically (not through UI), redirects should not happen
    await updateScopes(scopesService, ['grafana']);
    // Should stay on the current page, not redirect
    expect(locationService.getLocation().pathname).toBe('/');
    expect(fetchDashboardsSpy).toHaveBeenCalled();
  });

  it('Shows a proper message when no scopes are selected', async () => {
    await toggleDashboards();
    expectNoDashboardsNoScopes();
  });

  it('Does not show the input when there are no dashboards found for scope', async () => {
    await updateScopes(scopesService, ['cloud']);
    await toggleDashboards();
    expectNoDashboardsForScope();
  });

  it('Shows the input and a message when there are no dashboards found for filter', async () => {
    await updateScopes(scopesService, ['mimir']);
    await searchDashboards('unknown');
    expectDashboardsSearch();
    await waitFor(() => expectNoDashboardsForFilter());

    await clearNotFound();
    expectDashboardSearchValue('');
  });

  describe('groupDashboards', () => {
    it('Assigns dashboards without groups to root folder', () => {
      expect(scopesDashboardsService.groupSuggestedItems([dashboardWithoutFolder])).toEqual({
        '': {
          title: '',
          expanded: true,
          folders: {},
          suggestedNavigations: {
            [dashboardWithoutFolder.spec.dashboard]: {
              url: '/d/' + dashboardWithoutFolder.spec.dashboard,
              title: dashboardWithoutFolder.status.dashboardTitle,
              id: dashboardWithoutFolder.spec.dashboard,
            },
          },
        },
      });
    });

    it('Assigns dashboards with root group to root folder', () => {
      expect(scopesDashboardsService.groupSuggestedItems([dashboardWithRootFolder])).toEqual({
        '': {
          title: '',
          expanded: true,
          folders: {},
          suggestedNavigations: {
            [dashboardWithRootFolder.spec.dashboard]: {
              url: '/d/' + dashboardWithRootFolder.spec.dashboard,
              title: dashboardWithRootFolder.status.dashboardTitle,
              id: dashboardWithRootFolder.spec.dashboard,
            },
          },
        },
      });
    });

    it('Merges folders from multiple dashboards', () => {
      expect(scopesDashboardsService.groupSuggestedItems([dashboardWithOneFolder, dashboardWithTwoFolders])).toEqual({
        '': {
          title: '',
          expanded: true,
          folders: {
            'Folder 1': {
              title: 'Folder 1',
              expanded: false,
              folders: {},
              suggestedNavigations: {
                [dashboardWithOneFolder.spec.dashboard]: {
                  url: '/d/' + dashboardWithOneFolder.spec.dashboard,
                  title: dashboardWithOneFolder.status.dashboardTitle,
                  id: dashboardWithOneFolder.spec.dashboard,
                },
                [dashboardWithTwoFolders.spec.dashboard]: {
                  url: '/d/' + dashboardWithTwoFolders.spec.dashboard,
                  title: dashboardWithTwoFolders.status.dashboardTitle,
                  id: dashboardWithTwoFolders.spec.dashboard,
                },
              },
            },
            'Folder 2': {
              title: 'Folder 2',
              expanded: false,
              folders: {},
              suggestedNavigations: {
                [dashboardWithTwoFolders.spec.dashboard]: {
                  url: '/d/' + dashboardWithTwoFolders.spec.dashboard,
                  title: dashboardWithTwoFolders.status.dashboardTitle,
                  id: dashboardWithTwoFolders.spec.dashboard,
                },
              },
            },
          },
          suggestedNavigations: {},
        },
      });
    });

    it('Merges scopes from multiple dashboards', () => {
      expect(
        scopesDashboardsService.groupSuggestedItems([dashboardWithTwoFolders, alternativeDashboardWithTwoFolders])
      ).toEqual({
        '': {
          title: '',
          expanded: true,
          folders: {
            'Folder 1': {
              title: 'Folder 1',
              expanded: false,
              folders: {},
              suggestedNavigations: {
                [dashboardWithTwoFolders.spec.dashboard]: {
                  url: '/d/' + dashboardWithTwoFolders.spec.dashboard,
                  title: dashboardWithTwoFolders.status.dashboardTitle,
                  id: dashboardWithTwoFolders.spec.dashboard,
                },
              },
            },
            'Folder 2': {
              title: 'Folder 2',
              expanded: false,
              folders: {},
              suggestedNavigations: {
                [dashboardWithTwoFolders.spec.dashboard]: {
                  url: '/d/' + dashboardWithTwoFolders.spec.dashboard,
                  title: dashboardWithTwoFolders.status.dashboardTitle,
                  id: dashboardWithTwoFolders.spec.dashboard,
                },
              },
            },
          },
          suggestedNavigations: {},
        },
      });
    });

    it('Matches snapshot', () => {
      expect(
        scopesDashboardsService.groupSuggestedItems([
          dashboardWithoutFolder,
          dashboardWithOneFolder,
          dashboardWithTwoFolders,
          alternativeDashboardWithTwoFolders,
          dashboardWithRootFolder,
          alternativeDashboardWithRootFolder,
          dashboardWithRootFolderAndOtherFolder,
        ])
      ).toEqual({
        '': {
          suggestedNavigations: {
            [dashboardWithRootFolderAndOtherFolder.spec.dashboard]: {
              url: '/d/' + dashboardWithRootFolderAndOtherFolder.spec.dashboard,
              title: dashboardWithRootFolderAndOtherFolder.status.dashboardTitle,
              id: dashboardWithRootFolderAndOtherFolder.spec.dashboard,
            },
            [dashboardWithRootFolder.spec.dashboard]: {
              url: '/d/' + dashboardWithRootFolder.spec.dashboard,
              title: dashboardWithRootFolder.status.dashboardTitle,
              id: dashboardWithRootFolder.spec.dashboard,
            },
            [dashboardWithoutFolder.spec.dashboard]: {
              url: '/d/' + dashboardWithoutFolder.spec.dashboard,
              title: dashboardWithoutFolder.status.dashboardTitle,
              id: dashboardWithoutFolder.spec.dashboard,
            },
          },
          folders: {
            'Folder 1': {
              suggestedNavigations: {
                [dashboardWithOneFolder.spec.dashboard]: {
                  url: '/d/' + dashboardWithOneFolder.spec.dashboard,
                  title: dashboardWithOneFolder.status.dashboardTitle,
                  id: dashboardWithOneFolder.spec.dashboard,
                },
                [dashboardWithTwoFolders.spec.dashboard]: {
                  url: '/d/' + dashboardWithTwoFolders.spec.dashboard,
                  title: dashboardWithTwoFolders.status.dashboardTitle,
                  id: dashboardWithTwoFolders.spec.dashboard,
                },
              },
              folders: {},
              expanded: false,
              title: 'Folder 1',
            },
            'Folder 2': {
              suggestedNavigations: {
                [dashboardWithTwoFolders.spec.dashboard]: {
                  url: '/d/' + dashboardWithTwoFolders.spec.dashboard,
                  title: dashboardWithTwoFolders.status.dashboardTitle,
                  id: dashboardWithTwoFolders.spec.dashboard,
                },
              },
              folders: {},
              expanded: false,
              title: 'Folder 2',
            },
            'Folder 3': {
              suggestedNavigations: {
                [dashboardWithRootFolderAndOtherFolder.spec.dashboard]: {
                  url: '/d/' + dashboardWithRootFolderAndOtherFolder.spec.dashboard,
                  title: dashboardWithRootFolderAndOtherFolder.status.dashboardTitle,
                  id: dashboardWithRootFolderAndOtherFolder.spec.dashboard,
                },
              },
              folders: {},
              expanded: false,
              title: 'Folder 3',
            },
          },
          expanded: true,
          title: '',
        },
      });
    });
  });

  describe('subScopes', () => {
    beforeAll(() => {
      config.featureToggles.useScopesNavigationEndpoint = true;
    });

    afterAll(() => {
      config.featureToggles.useScopesNavigationEndpoint = false;
    });

    it('Creates subScope folders when navigation items have subScope', async () => {
      const mockNavigations = [navigationWithSubScope, navigationWithSubScopeDifferent];
      fetchScopeNavigationsSpy.mockResolvedValue(mockNavigations);

      await toggleDashboards();
      await updateScopes(scopesService, ['grafana']);
      await jest.runOnlyPendingTimersAsync();

      // Verify subScope folders are created
      expect(screen.getByTestId('scopes-dashboards-Mimir Dashboards-expand')).toBeInTheDocument();
      expect(screen.getByTestId('scopes-dashboards-Loki Dashboards-expand')).toBeInTheDocument();
    });

    it('Loads subScope items when folder is expanded', async () => {
      const mockNavigations = [navigationWithSubScope];
      fetchScopeNavigationsSpy.mockResolvedValueOnce(mockNavigations).mockResolvedValueOnce(subScopeMimirItems);

      await toggleDashboards();
      await updateScopes(scopesService, ['grafana']);
      await jest.runOnlyPendingTimersAsync();

      // Verify folder appears
      expect(screen.getByTestId('scopes-dashboards-Mimir Dashboards-expand')).toBeInTheDocument();

      // Expand the subScope folder
      await expandDashboardFolder('Mimir Dashboards');

      // Wait for async fetchSubScopeItems to complete
      await waitFor(() => {
        expect(fetchScopeNavigationsSpy).toHaveBeenCalledWith(['mimir']);
      });
      await jest.runOnlyPendingTimersAsync();

      // Items are added to nested folders within the subScope folder, so expand those folders
      await expandDashboardFolder('General');
      await expandDashboardFolder('Observability');
      await jest.runOnlyPendingTimersAsync();

      // Verify loaded content appears (IDs are based on metadata.name)
      await waitFor(() => {
        expectDashboardInDocument('mimir-item-1');
      });
      expectDashboardInDocument('mimir-item-2');
    });

    it('Shows loading state while fetching subScope items', async () => {
      const mockNavigations = [navigationWithSubScope];
      fetchScopeNavigationsSpy.mockResolvedValueOnce(mockNavigations).mockResolvedValueOnce(subScopeMimirItems);

      await toggleDashboards();
      await updateScopes(scopesService, ['grafana']);
      await jest.runOnlyPendingTimersAsync();

      // Verify folder appears
      expect(screen.getByTestId('scopes-dashboards-Mimir Dashboards-expand')).toBeInTheDocument();

      // Expand the subScope folder
      await expandDashboardFolder('Mimir Dashboards');

      // Verify fetch was called (loading happens asynchronously)
      await waitFor(() => {
        expect(fetchScopeNavigationsSpy).toHaveBeenCalledWith(['mimir']);
      });
    });

    it('Multiple subScope folders with same subScope load same content', async () => {
      const mockNavigations = [navigationWithSubScope, navigationWithSubScope2];
      fetchScopeNavigationsSpy.mockResolvedValueOnce(mockNavigations).mockResolvedValue(subScopeMimirItems);

      await toggleDashboards();
      await updateScopes(scopesService, ['grafana']);
      await jest.runOnlyPendingTimersAsync();

      // Verify folders appear
      expect(screen.getByTestId('scopes-dashboards-Mimir Dashboards-expand')).toBeInTheDocument();
      expect(screen.getByTestId('scopes-dashboards-Mimir Overview-expand')).toBeInTheDocument();

      // Expand first subScope folder
      await expandDashboardFolder('Mimir Dashboards');

      // Wait for fetch to complete
      await waitFor(() => {
        expect(fetchScopeNavigationsSpy).toHaveBeenCalledWith(['mimir']);
      });
      await jest.runOnlyPendingTimersAsync();

      // Expand nested folders to see the content in first subScope folder
      await expandDashboardFolder('General');
      await expandDashboardFolder('Observability');
      await jest.runOnlyPendingTimersAsync();

      // Verify content appears in first folder
      await waitFor(() => {
        expectDashboardInDocument('mimir-item-1');
      });
      expectDashboardInDocument('mimir-item-2');

      // Expand second subScope folder (same subScope) - it should load the same content
      await expandDashboardFolder('Mimir Overview');

      // Wait for fetch to complete (should use cached data or fetch again)
      await waitFor(() => {
        // The fetch might be called again or might use cached content
        expect(fetchScopeNavigationsSpy.mock.calls.length).toBeGreaterThanOrEqual(1);
      });
      await jest.runOnlyPendingTimersAsync();

      // Both folders should have the same content (IDs are based on metadata.name)
      // Since nested folders are already expanded, content should be visible
      expectDashboardInDocument('mimir-item-1');
      expectDashboardInDocument('mimir-item-2');
    });

    it('Handles errors when fetching subScope items', async () => {
      const mockNavigations = [navigationWithSubScope];
      fetchScopeNavigationsSpy.mockResolvedValueOnce(mockNavigations).mockRejectedValueOnce(new Error('Fetch failed'));

      await toggleDashboards();
      await updateScopes(scopesService, ['grafana']);
      await jest.runOnlyPendingTimersAsync();

      // Verify folder appears
      expect(screen.getByTestId('scopes-dashboards-Mimir Dashboards-expand')).toBeInTheDocument();

      // Expand the subScope folder
      await expandDashboardFolder('Mimir Dashboards');
      await jest.runOnlyPendingTimersAsync();

      // Verify fetch was called
      expect(fetchScopeNavigationsSpy).toHaveBeenCalledWith(['mimir']);

      // Verify no content appears (error handled gracefully)
      expectDashboardNotInDocument('mimir-item-1');
    });

    it('Ignores groups for subScope items', async () => {
      const mockNavigations = [navigationWithSubScopeAndGroups];
      fetchScopeNavigationsSpy.mockResolvedValue(mockNavigations);

      await toggleDashboards();
      await updateScopes(scopesService, ['grafana']);
      await jest.runOnlyPendingTimersAsync();

      // Verify subScope folder is created (groups should be ignored)
      expect(screen.getByTestId('scopes-dashboards-Mimir with Groups-expand')).toBeInTheDocument();

      // The folder should exist regardless of groups
      expect(fetchScopeNavigationsSpy).toHaveBeenCalled();
    });

    it('Filters search works with loaded subScope content', async () => {
      const mockNavigations = [navigationWithSubScope];
      fetchScopeNavigationsSpy.mockResolvedValueOnce(mockNavigations).mockResolvedValueOnce(subScopeMimirItems);

      await toggleDashboards();
      await updateScopes(scopesService, ['grafana']);
      await jest.runOnlyPendingTimersAsync();

      // Verify folder appears
      expect(screen.getByTestId('scopes-dashboards-Mimir Dashboards-expand')).toBeInTheDocument();

      // Expand subScope folder and load content
      await expandDashboardFolder('Mimir Dashboards');

      // Wait for fetch to complete
      await waitFor(() => {
        expect(fetchScopeNavigationsSpy).toHaveBeenCalledWith(['mimir']);
      });
      await jest.runOnlyPendingTimersAsync();

      // Expand nested folders to see the content
      await expandDashboardFolder('General');
      await expandDashboardFolder('Observability');
      await jest.runOnlyPendingTimersAsync();

      // Verify content is loaded
      await waitFor(() => {
        expectDashboardInDocument('mimir-item-1');
      });
      expectDashboardInDocument('mimir-item-2');

      // Search for a dashboard in the subScope
      await searchDashboards('Mimir Dashboard 1');

      // Verify search works (IDs are based on metadata.name)
      expectDashboardInDocument('mimir-item-1');
      expectDashboardNotInDocument('mimir-item-2');
    });

    it('Does not fetch subScope items if folder is already loaded', async () => {
      const mockNavigations = [navigationWithSubScope];
      fetchScopeNavigationsSpy.mockResolvedValueOnce(mockNavigations).mockResolvedValueOnce(subScopeMimirItems);

      await toggleDashboards();
      await updateScopes(scopesService, ['grafana']);
      await jest.runOnlyPendingTimersAsync();

      // Verify folder appears
      expect(screen.getByTestId('scopes-dashboards-Mimir Dashboards-expand')).toBeInTheDocument();

      // Expand the subScope folder first time
      await expandDashboardFolder('Mimir Dashboards');

      // Wait for fetch to complete
      await waitFor(() => {
        expect(fetchScopeNavigationsSpy).toHaveBeenCalledWith(['mimir']);
      });
      await jest.runOnlyPendingTimersAsync();

      // Expand nested folders to see the content
      await expandDashboardFolder('General');
      await jest.runOnlyPendingTimersAsync();

      // Verify content is loaded
      await waitFor(() => {
        expectDashboardInDocument('mimir-item-1');
      });

      const firstCallCount = fetchScopeNavigationsSpy.mock.calls.length;

      // Collapse and expand again
      await expandDashboardFolder('Mimir Dashboards'); // Collapse
      await expandDashboardFolder('Mimir Dashboards'); // Expand again
      await jest.runOnlyPendingTimersAsync();

      // Should not fetch again if already loaded
      // Note: This test might need adjustment based on actual implementation
      expect(fetchScopeNavigationsSpy.mock.calls.length).toBeGreaterThanOrEqual(firstCallCount);
    });
  });

  describe('filterFolders', () => {
    it('Shows folders matching criteria', () => {
      expect(
        scopesDashboardsService.filterFolders(
          {
            '': {
              title: '',
              expanded: true,
              folders: {
                'Folder 1': {
                  title: 'Folder 1',
                  expanded: false,
                  folders: {},
                  suggestedNavigations: {
                    'Dashboard ID': {
                      url: '/d/Dashboard ID',
                      title: 'Dashboard Title',
                      id: 'Dashboard ID',
                    },
                  },
                },
                'Folder 2': {
                  title: 'Folder 2',
                  expanded: true,
                  folders: {},
                  suggestedNavigations: {
                    'Dashboard ID': {
                      url: '/d/Dashboard ID',
                      title: 'Dashboard Title',
                      id: 'Dashboard ID',
                    },
                  },
                },
              },
              suggestedNavigations: {
                'Dashboard ID': {
                  url: '/d/Dashboard ID',
                  title: 'Dashboard Title',
                  id: 'Dashboard ID',
                },
              },
            },
          },
          'Folder'
        )
      ).toEqual({
        '': {
          title: '',
          expanded: true,
          folders: {
            'Folder 1': {
              title: 'Folder 1',
              expanded: true,
              folders: {},
              suggestedNavigations: {
                'Dashboard ID': {
                  url: '/d/Dashboard ID',
                  title: 'Dashboard Title',
                  id: 'Dashboard ID',
                },
              },
            },
            'Folder 2': {
              title: 'Folder 2',
              expanded: true,
              folders: {},
              suggestedNavigations: {
                'Dashboard ID': {
                  url: '/d/Dashboard ID',
                  title: 'Dashboard Title',
                  id: 'Dashboard ID',
                },
              },
            },
          },
          suggestedNavigations: {},
        },
      });
    });

    it('Shows dashboards matching criteria', () => {
      expect(
        scopesDashboardsService.filterFolders(
          {
            '': {
              title: '',
              expanded: true,
              folders: {
                'Folder 1': {
                  title: 'Folder 1',
                  expanded: false,
                  folders: {},
                  suggestedNavigations: {
                    'Dashboard ID': {
                      url: '/d/Dashboard ID',
                      title: 'Dashboard Title',
                      id: 'Dashboard ID',
                    },
                  },
                },
                'Folder 2': {
                  title: 'Folder 2',
                  expanded: true,
                  folders: {},
                  suggestedNavigations: {
                    'Random ID': {
                      url: '/d/Random ID',
                      title: 'Random Title',
                      id: 'Random ID',
                    },
                  },
                },
              },
              suggestedNavigations: {
                'Dashboard ID': {
                  url: '/d/Dashboard ID',
                  title: 'Dashboard Title',
                  id: 'Dashboard ID',
                },
                'Random ID': {
                  url: '/d/Random ID',
                  title: 'Random Title',
                  id: 'Random ID',
                },
              },
            },
          },
          'dash'
        )
      ).toEqual({
        '': {
          title: '',
          expanded: true,
          folders: {
            'Folder 1': {
              title: 'Folder 1',
              expanded: true,
              folders: {},
              suggestedNavigations: {
                'Dashboard ID': {
                  url: '/d/Dashboard ID',
                  title: 'Dashboard Title',
                  id: 'Dashboard ID',
                },
              },
            },
          },
          suggestedNavigations: {
            'Dashboard ID': {
              url: '/d/Dashboard ID',
              title: 'Dashboard Title',
              id: 'Dashboard ID',
            },
          },
        },
      });
    });
  });
});
