import { useState, useEffect } from 'react';

import { VizPanel, UrlSyncManager } from '@grafana/scenes';

import { DashboardScene } from '../scene/DashboardScene';
import { DashboardRepeatsProcessedEvent } from '../scene/types/DashboardRepeatsProcessedEvent';
import { containsPathIdSeparator, findVizPanelByPathId } from '../utils/pathId';

export function useSoloPanel(dashboard: DashboardScene, pathId: string): [VizPanel | undefined, string | undefined] {
  const [panel, setPanel] = useState<VizPanel>();
  const [error, setError] = useState<string | undefined>();

  useEffect(() => {
    const urlSyncManager = new UrlSyncManager();
    urlSyncManager.initSync(dashboard);

    const cleanUp = dashboard.activate();

    let panel: VizPanel | null = null;
    try {
      panel = findVizPanelByPathId(dashboard, pathId);
    } catch (e) {
      // do nothing, just the panel is not found or not a VizPanel
    }

    if (panel) {
      activateParents(panel);
      setPanel(panel);
    } else if (containsPathIdSeparator(pathId)) {
      findRepeatClone(dashboard, pathId).then((panel) => {
        if (panel) {
          setPanel(panel);
        } else {
          setError('Panel not found');
        }
      });
    } else {
      setError('Panel not found');
    }

    return cleanUp;
  }, [dashboard, pathId]);

  return [panel, error];
}

function activateParents(panel: VizPanel) {
  let parent = panel.parent;

  while (parent && !parent.isActive) {
    parent.activate();
    parent = parent.parent;
  }
}

function findRepeatClone(dashboard: DashboardScene, pathId: string): Promise<VizPanel | undefined> {
  return new Promise((resolve) => {
    dashboard.subscribeToEvent(DashboardRepeatsProcessedEvent, () => {
      const panel = findVizPanelByPathId(dashboard, pathId);
      if (panel) {
        resolve(panel);
      } else {
        // If rows are repeated they could add new panel repeaters that needs to be activated
        dashboard.state.body.activateRepeaters?.();
      }
    });

    dashboard.state.body.activateRepeaters?.();
  });
}
