import { jsx, jsxs, Fragment } from 'react/jsx-runtime';
import * as React from 'react';
import { useRef } from 'react';
import { selectors } from '@grafana/e2e-selectors';
import { t } from '@grafana/i18n';
import { EditorField } from '@grafana/plugin-ui';
import { AutoSizeInput, Select } from '@grafana/ui';
import { LegendFormatMode } from '../../types.mjs';

const getLegendModeOptions = () => [
  {
    label: t("grafana-prometheus.prom-query-legend-editor.get-legend-mode-options.label-auto", "Auto"),
    value: LegendFormatMode.Auto,
    description: t(
      "grafana-prometheus.prom-query-legend-editor.get-legend-mode-options.description-auto",
      "Only includes unique labels"
    )
  },
  {
    label: t("grafana-prometheus.prom-query-legend-editor.get-legend-mode-options.label-verbose", "Verbose"),
    value: LegendFormatMode.Verbose,
    description: t(
      "grafana-prometheus.prom-query-legend-editor.get-legend-mode-options.description-verbose",
      "All label names and values"
    )
  },
  {
    label: t("grafana-prometheus.prom-query-legend-editor.get-legend-mode-options.label-custom", "Custom"),
    value: LegendFormatMode.Custom,
    description: t(
      "grafana-prometheus.prom-query-legend-editor.get-legend-mode-options.description-custom",
      "Provide a naming template"
    )
  }
];
const PromQueryLegendEditor = React.memo(
  ({ legendFormat, onChange, onRunQuery }) => {
    const mode = getLegendMode(legendFormat);
    const inputRef = useRef(null);
    const legendModeOptions = getLegendModeOptions();
    const onLegendFormatChanged = (evt) => {
      let newFormat = evt.currentTarget.value;
      if (newFormat.length === 0) {
        newFormat = LegendFormatMode.Auto;
      }
      if (newFormat !== legendFormat) {
        onChange(newFormat);
        onRunQuery();
      }
    };
    const onLegendModeChanged = (value) => {
      switch (value.value) {
        case LegendFormatMode.Auto:
          onChange(LegendFormatMode.Auto);
          break;
        case LegendFormatMode.Custom:
          onChange("{{label_name}}");
          setTimeout(() => {
            var _a, _b;
            (_a = inputRef.current) == null ? void 0 : _a.focus();
            (_b = inputRef.current) == null ? void 0 : _b.setSelectionRange(2, 12, "forward");
          }, 10);
          break;
        case LegendFormatMode.Verbose:
          onChange("");
          break;
      }
      onRunQuery();
    };
    return /* @__PURE__ */ jsx(
      EditorField,
      {
        label: t("grafana-prometheus.querybuilder.prom-query-legend-editor.label-legend", "Legend"),
        tooltip: t(
          "grafana-prometheus.querybuilder.prom-query-legend-editor.tooltip-legend",
          "Series name override or template. Ex. {{templateExample}} will be replaced with label value for {{labelName}}.",
          { templateExample: "{{hostname}}", labelName: "hostname" }
        ),
        "data-testid": selectors.components.DataSource.Prometheus.queryEditor.legend,
        children: /* @__PURE__ */ jsxs(Fragment, { children: [
          mode === LegendFormatMode.Custom && /* @__PURE__ */ jsx(
            AutoSizeInput,
            {
              id: "legendFormat",
              minWidth: 22,
              placeholder: "auto",
              defaultValue: legendFormat,
              onCommitChange: onLegendFormatChanged,
              ref: inputRef
            }
          ),
          mode !== LegendFormatMode.Custom && /* @__PURE__ */ jsx(
            Select,
            {
              inputId: "legend.mode",
              isSearchable: false,
              placeholder: t(
                "grafana-prometheus.querybuilder.prom-query-legend-editor.placeholder-select-legend-mode",
                "Select legend mode"
              ),
              options: legendModeOptions,
              width: 22,
              onChange: onLegendModeChanged,
              value: legendModeOptions.find((x) => x.value === mode)
            }
          )
        ] })
      }
    );
  }
);
PromQueryLegendEditor.displayName = "PromQueryLegendEditor";
function getLegendMode(legendFormat) {
  if (legendFormat === LegendFormatMode.Auto) {
    return LegendFormatMode.Auto;
  }
  if (legendFormat == null || legendFormat === "") {
    return LegendFormatMode.Verbose;
  }
  return LegendFormatMode.Custom;
}
function getLegendModeLabel(legendFormat) {
  var _a;
  const legendModeOptions = getLegendModeOptions();
  const mode = getLegendMode(legendFormat);
  if (mode !== LegendFormatMode.Custom) {
    return (_a = legendModeOptions.find((x) => x.value === mode)) == null ? void 0 : _a.label;
  }
  return legendFormat;
}

export { PromQueryLegendEditor, getLegendModeLabel };
//# sourceMappingURL=PromQueryLegendEditor.mjs.map
