import { jsx, jsxs } from 'react/jsx-runtime';
import { getMinMaxAndDelta, isDataFrameWithValue, formattedValueToString, FieldType, isDataFrame } from '@grafana/data';
import { TableCellDisplayMode, GraphDrawStyle, LineInterpolation, GraphGradientMode, BarAlignment, VisibilityMode } from '@grafana/schema';
import { useTheme2 } from '../../../themes/ThemeContext.mjs';
import { measureText } from '../../../utils/measureText.mjs';
import { FormattedValueDisplay } from '../../FormattedValueDisplay/FormattedValueDisplay.mjs';
import { Sparkline } from '../../Sparkline/Sparkline.mjs';
import { getAlignmentFactor, getCellOptions } from '../utils.mjs';

const defaultSparklineCellConfig = {
  type: TableCellDisplayMode.Sparkline,
  drawStyle: GraphDrawStyle.Line,
  lineInterpolation: LineInterpolation.Smooth,
  lineWidth: 1,
  fillOpacity: 17,
  gradientMode: GraphGradientMode.Hue,
  pointSize: 2,
  barAlignment: BarAlignment.Center,
  showPoints: VisibilityMode.Never,
  hideValue: false
};
const SparklineCell = (props) => {
  var _a, _b;
  const { field, innerWidth, tableStyles, cell, cellProps, timeRange } = props;
  const sparkline = getSparkline(cell.value);
  const theme = useTheme2();
  if (!sparkline) {
    return /* @__PURE__ */ jsx("div", { ...cellProps, className: tableStyles.cellContainer, children: field.config.noValue || "no data" });
  }
  if (sparkline.x && !sparkline.x.config.interval && sparkline.x.values.length > 1) {
    sparkline.x.config.interval = sparkline.x.values[1] - sparkline.x.values[0];
  }
  sparkline.y.values = sparkline.y.values.map((v) => {
    if (!Number.isFinite(v)) {
      return null;
    } else {
      return v;
    }
  });
  const range = getMinMaxAndDelta(sparkline.y);
  sparkline.y.config.min = range.min;
  sparkline.y.config.max = range.max;
  sparkline.y.state = { range };
  sparkline.timeRange = timeRange;
  const cellOptions = getTableSparklineCellOptions(field);
  const config = {
    color: field.config.color,
    custom: {
      ...defaultSparklineCellConfig,
      ...cellOptions
    }
  };
  const hideValue = (_b = (_a = field.config.custom) == null ? void 0 : _a.cellOptions) == null ? void 0 : _b.hideValue;
  let valueWidth = 0;
  let valueElement = null;
  if (!hideValue) {
    const value = isDataFrameWithValue(cell.value) ? cell.value.value : null;
    const displayValue = field.display(value);
    const alignmentFactor = getAlignmentFactor(field, displayValue, cell.row.index);
    valueWidth = measureText(formattedValueToString(alignmentFactor), 16).width + theme.spacing.gridSize;
    valueElement = /* @__PURE__ */ jsx(
      FormattedValueDisplay,
      {
        style: {
          width: `${valueWidth - theme.spacing.gridSize}px`,
          textAlign: "right",
          marginRight: theme.spacing(1)
        },
        value: displayValue
      }
    );
  }
  return /* @__PURE__ */ jsxs("div", { ...cellProps, className: tableStyles.cellContainer, children: [
    valueElement,
    /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx(
      Sparkline,
      {
        width: innerWidth - valueWidth,
        height: tableStyles.cellHeightInner,
        sparkline,
        config,
        theme: tableStyles.theme
      }
    ) })
  ] });
};
function getSparkline(value) {
  if (Array.isArray(value)) {
    return {
      y: {
        name: "test",
        type: FieldType.number,
        values: value,
        config: {}
      }
    };
  }
  if (isDataFrame(value)) {
    const timeField = value.fields.find((x) => x.type === FieldType.time);
    const numberField = value.fields.find((x) => x.type === FieldType.number);
    if (timeField && numberField) {
      return { x: timeField, y: numberField };
    }
  }
  return;
}
function getTableSparklineCellOptions(field) {
  let options = getCellOptions(field);
  if (options.type === TableCellDisplayMode.Auto) {
    options = { ...options, type: TableCellDisplayMode.Sparkline };
  }
  if (options.type === TableCellDisplayMode.Sparkline) {
    return options;
  }
  throw new Error(`Expected options type ${TableCellDisplayMode.Sparkline} but got ${options.type}`);
}

export { SparklineCell, defaultSparklineCellConfig };
//# sourceMappingURL=SparklineCell.mjs.map
