import { jsxs, Fragment, jsx } from 'react/jsx-runtime';
import { useCallback } from 'react';
import { getDebounceTimeInMilliseconds } from '../../caching.mjs';
import { truncateResult } from '../../language_utils.mjs';
import { regexifyLabelValuesQueryString } from '../parsingUtils.mjs';
import { promQueryModeller } from '../shared/modeller_instance.mjs';
import { LabelFilters } from './LabelFilters.mjs';
import { MetricCombobox } from './MetricCombobox.mjs';

function MetricsLabelsSection({
  datasource,
  query,
  onChange,
  onBlur,
  variableEditor,
  timeRange
}) {
  const onChangeLabels = (labels) => {
    onChange({ ...query, labels });
  };
  const withTemplateVariableOptions = useCallback(
    async (optionsPromise) => {
      const variables = datasource.getVariables();
      const options = await optionsPromise;
      return [
        ...variables.map((value) => ({ label: value, value })),
        ...options.map((option) => ({
          label: option.value,
          value: option.value,
          title: option.description
        }))
      ];
    },
    [datasource]
  );
  const onGetLabelNames = async (forLabel) => {
    if (!query.metric) {
      await datasource.languageProvider.queryLabelKeys(timeRange);
      return datasource.languageProvider.retrieveLabelKeys().map((k) => ({ value: k }));
    }
    const labelsToConsider = query.labels.filter((x) => x !== forLabel);
    labelsToConsider.push({ label: "__name__", op: "=", value: query.metric });
    const expr = promQueryModeller.renderLabels(labelsToConsider);
    let labelsIndex = await datasource.languageProvider.queryLabelKeys(timeRange, expr);
    return labelsIndex.filter((labelName) => !labelsToConsider.find((filter) => filter.label === labelName)).map((k) => ({ value: k }));
  };
  const getLabelValuesAutocompleteSuggestions = async (queryString, labelName) => {
    const forLabel = {
      label: labelName != null ? labelName : "__name__",
      op: "=~",
      value: regexifyLabelValuesQueryString(`.*${queryString}`)
    };
    const labelsToConsider = query.labels.filter((x) => x.label !== forLabel.label);
    labelsToConsider.push(forLabel);
    if (query.metric) {
      labelsToConsider.push({ label: "__name__", op: "=", value: query.metric });
    }
    const interpolatedLabelsToConsider = labelsToConsider.map((labelObject) => ({
      ...labelObject,
      label: datasource.interpolateString(labelObject.label),
      value: datasource.interpolateString(labelObject.value)
    }));
    const expr = promQueryModeller.renderLabels(interpolatedLabelsToConsider);
    const values = await datasource.languageProvider.queryLabelValues(timeRange, forLabel.label, expr);
    return truncateResult(values).map(toSelectableValue);
  };
  const onGetLabelValues = async (forLabel) => {
    if (!forLabel.label) {
      return [];
    }
    if (!query.metric) {
      return (await datasource.languageProvider.queryLabelValues(timeRange, forLabel.label)).map((v) => ({ value: v }));
    }
    const labelsToConsider = query.labels.filter((x) => x !== forLabel);
    labelsToConsider.push({ label: "__name__", op: "=", value: query.metric });
    const interpolatedLabelsToConsider = labelsToConsider.map((labelObject) => ({
      ...labelObject,
      label: datasource.interpolateString(labelObject.label),
      value: datasource.interpolateString(labelObject.value)
    }));
    const expr = promQueryModeller.renderLabels(interpolatedLabelsToConsider);
    return (await datasource.languageProvider.queryLabelValues(timeRange, forLabel.label, expr)).map(toSelectableValue);
  };
  const onGetMetrics = useCallback(() => {
    return withTemplateVariableOptions(getMetrics(datasource, query, timeRange));
  }, [datasource, query, timeRange, withTemplateVariableOptions]);
  return /* @__PURE__ */ jsxs(Fragment, { children: [
    /* @__PURE__ */ jsx(
      MetricCombobox,
      {
        query,
        onChange,
        onGetMetrics,
        datasource,
        labelsFilters: query.labels,
        metricLookupDisabled: datasource.lookupsDisabled,
        onBlur: onBlur ? onBlur : () => {
        },
        variableEditor,
        timeRange
      }
    ),
    /* @__PURE__ */ jsx(
      LabelFilters,
      {
        debounceDuration: getDebounceTimeInMilliseconds(datasource.cacheLevel),
        getLabelValuesAutofillSuggestions: getLabelValuesAutocompleteSuggestions,
        labelsFilters: query.labels,
        onChange: onChangeLabels,
        onGetLabelNames: (forLabel) => withTemplateVariableOptions(onGetLabelNames(forLabel)),
        onGetLabelValues: (forLabel) => withTemplateVariableOptions(onGetLabelValues(forLabel)),
        variableEditor
      }
    )
  ] });
}
async function getMetrics(datasource, query, timeRange) {
  var _a;
  const metadata = datasource.languageProvider.retrieveMetricsMetadata();
  if (Object.keys(metadata).length === 0) {
    await datasource.languageProvider.queryMetricsMetadata();
  }
  let metrics;
  const expr = promQueryModeller.renderLabels(query.labels);
  metrics = (_a = await datasource.languageProvider.queryLabelValues(timeRange, "__name__", expr === "" ? void 0 : expr)) != null ? _a : [];
  return metrics.map((m) => ({
    value: m,
    description: getMetadataString(m, datasource.languageProvider.retrieveMetricsMetadata())
  }));
}
function getMetadataString(metric, metadata) {
  if (!metadata[metric]) {
    return;
  }
  const { type, help } = metadata[metric];
  return `${type.toUpperCase()}: ${help}`;
}
function toSelectableValue(lv) {
  return {
    label: lv,
    value: lv
  };
}

export { MetricsLabelsSection, getMetadataString };
//# sourceMappingURL=MetricsLabelsSection.mjs.map
