import { jsxs, jsx, Fragment } from 'react/jsx-runtime';
import { css } from '@emotion/css';
import { useMemo, useCallback } from 'react';
import { FieldMatcherID, fieldMatchers, fieldReducers, ReducerID } from '@grafana/data';
import { t } from '@grafana/i18n';
import { ComparisonOperation } from '@grafana/schema';
import { useStyles2 } from '../../themes/ThemeContext.mjs';
import { Input } from '../Input/Input.mjs';
import { Select } from '../Select/Select.mjs';

const comparisonOperationOptions = [
  { label: "==", value: ComparisonOperation.EQ },
  { label: "!=", value: ComparisonOperation.NEQ },
  { label: ">", value: ComparisonOperation.GT },
  { label: ">=", value: ComparisonOperation.GTE },
  { label: "<", value: ComparisonOperation.LT },
  { label: "<=", value: ComparisonOperation.LTE }
];
function isBooleanReducer(r) {
  return r === ReducerID.allIsNull || r === ReducerID.allIsZero;
}
const FieldValueMatcherEditor = ({ options, onChange }) => {
  const styles = useStyles2(getStyles);
  const reducer = useMemo(() => fieldReducers.selectOptions([options == null ? void 0 : options.reducer]), [options == null ? void 0 : options.reducer]);
  const onSetReducer = useCallback(
    (selection) => {
      return onChange({ ...options, reducer: selection.value });
    },
    [options, onChange]
  );
  const onChangeOp = useCallback(
    (v) => {
      return onChange({ ...options, op: v.value });
    },
    [options, onChange]
  );
  const onChangeValue = useCallback(
    (e) => {
      const value = e.currentTarget.valueAsNumber;
      return onChange({ ...options, value });
    },
    [options, onChange]
  );
  const opts = options != null ? options : {};
  const isBool = isBooleanReducer(opts.reducer);
  return /* @__PURE__ */ jsxs("div", { className: styles.spot, children: [
    /* @__PURE__ */ jsx(
      Select,
      {
        value: reducer.current,
        options: reducer.options,
        onChange: onSetReducer,
        placeholder: t("grafana-ui.field-value-matcher.select-field-placeholder", "Select field reducer")
      }
    ),
    opts.reducer && !isBool && /* @__PURE__ */ jsxs(Fragment, { children: [
      /* @__PURE__ */ jsx(
        Select,
        {
          value: comparisonOperationOptions.find((v) => v.value === opts.op),
          options: comparisonOperationOptions,
          onChange: onChangeOp,
          "aria-label": t("grafana-ui.field-value-matcher.operator-label", "Comparison operator"),
          width: 19
        }
      ),
      /* @__PURE__ */ jsx(Input, { type: "number", value: opts.value, onChange: onChangeValue })
    ] })
  ] });
};
const getStyles = (theme) => {
  return {
    spot: css({
      display: "flex",
      flexDirection: "row",
      alignItems: "center",
      alignContent: "flex-end",
      gap: "4px"
    })
  };
};
const getFieldValueMatcherItem = () => ({
  id: FieldMatcherID.byValue,
  component: FieldValueMatcherEditor,
  matcher: fieldMatchers.get(FieldMatcherID.byValue),
  name: t("grafana-ui.matchers-ui.name-fields-with-value", "Fields with values"),
  description: t(
    "grafana-ui.matchers-ui.description-fields-with-value",
    "Set properties for fields with reducer condition"
  ),
  optionsToLabel: (options) => `${options == null ? void 0 : options.reducer} ${options == null ? void 0 : options.op} ${options == null ? void 0 : options.value}`
});

export { FieldValueMatcherEditor, comparisonOperationOptions, getFieldValueMatcherItem };
//# sourceMappingURL=FieldValueMatcher.mjs.map
