import { css } from '@emotion/css';
import { useCallback } from 'react';

import { GrafanaTheme2 } from '@grafana/data';
import { t } from '@grafana/i18n';
import { Button, Dropdown, Menu, useStyles2 } from '@grafana/ui';
import { InspectTab } from 'app/features/inspector/types';

import { PanelInspectDrawer } from '../../../../inspect/PanelInspectDrawer';
import { getDashboardSceneFor } from '../../../../utils/utils';
import { QUERY_EDITOR_TYPE_CONFIG, QueryEditorType } from '../../constants';
import { useActionsContext, usePanelContext, useQueryEditorUIContext } from '../QueryEditorContext';

export function QueryActionsMenu() {
  const { duplicateQuery } = useActionsContext();
  const { panel } = usePanelContext();
  const {
    selectedQuery,
    selectedQueryDsData,
    selectedQueryDsLoading,
    showingDatasourceHelp,
    toggleDatasourceHelp,
    cardType,
  } = useQueryEditorUIContext();

  const styles = useStyles2(getStyles);

  const onOpenInspector = useCallback(() => {
    const dashboard = getDashboardSceneFor(panel);
    dashboard.showModal(new PanelInspectDrawer({ panelRef: panel.getRef(), currentTab: InspectTab.Query }));
  }, [panel]);

  if (!selectedQuery) {
    return null;
  }

  const typeLabel = QUERY_EDITOR_TYPE_CONFIG[cardType].getLabel();
  const isExpression = cardType === QueryEditorType.Expression;
  const hasEditorHelp = !selectedQueryDsLoading && selectedQueryDsData?.datasource?.components?.QueryEditorHelp;

  return (
    <Dropdown
      overlay={
        <Menu>
          <Menu.Item
            className={styles.menuItem}
            label={t('query-editor-next.action.duplicate', 'Duplicate {{type}}', { type: typeLabel })}
            icon="copy"
            onClick={() => duplicateQuery(selectedQuery.refId)}
          />

          {/* Data source help (queries only, not expressions) */}
          {hasEditorHelp && !isExpression && (
            <Menu.Item
              className={styles.menuItem}
              label={
                showingDatasourceHelp
                  ? t('query-editor-next.action.hide-help', 'Hide data source help')
                  : t('query-editor-next.action.show-help', 'Show data source help')
              }
              icon="question-circle"
              onClick={toggleDatasourceHelp}
              active={showingDatasourceHelp}
            />
          )}

          <Menu.Item
            className={styles.menuItem}
            label={t('query-editor-next.action.inspector', 'Query inspector')}
            icon="brackets-curly"
            onClick={onOpenInspector}
          />
        </Menu>
      }
      placement="bottom-end"
    >
      <Button
        size="sm"
        fill="text"
        icon="ellipsis-v"
        variant="secondary"
        aria-label={t('query-editor-next.action.more-actions', 'More {{type}} actions', { type: typeLabel })}
        tooltip={t('query-editor-next.action.more-actions', 'More {{type}} actions', { type: typeLabel })}
      />
    </Dropdown>
  );
}

const getStyles = (theme: GrafanaTheme2) => ({
  menuItem: css({
    fontSize: theme.typography.bodySmall.fontSize,
  }),
});
