import { Keeper } from 'app/extensions/api/clients/secret/v1beta1/endpoints.gen';

import { keeperToListItem } from './utils';

const createApiKeeper = (name: string, spec: Keeper['spec'], creationTimestamp = '2025-01-01T00:00:00Z'): Keeper =>
  ({
    metadata: { name, creationTimestamp },
    spec,
    status: {},
  }) as Keeper;

describe('keeperToListItem', () => {
  it('converts AWS keeper to list item', () => {
    const keeper = createApiKeeper('aws-test', {
      description: 'Test AWS',
      aws: { region: 'us-east-1' },
    });
    const listItem = keeperToListItem(keeper);

    expect(listItem.name).toBe('aws-test');
    expect(listItem.type).toBe('aws');
    expect(listItem.description).toBe('Test AWS');
    expect(listItem.isActive).toBe(false);
    expect(listItem.config).toBe('us-east-1');
  });

  it('defaults to system type when no provider config is set', () => {
    const keeper = createApiKeeper('system-test', {
      description: 'System keeper',
    });
    const listItem = keeperToListItem(keeper);

    expect(listItem.type).toBe('system');
    expect(listItem.config).toBe('');
  });

  it('preserves creation timestamp', () => {
    const keeper = createApiKeeper('test', {
      description: 'Test',
      aws: { region: 'us-east-1' },
    });
    const listItem = keeperToListItem(keeper);

    expect(listItem.createdAt).toBe('2025-01-01T00:00:00Z');
  });

  it('maps unhandled provider config to unknown type', () => {
    // Simulates a future provider key added to KeeperSpec by backend codegen
    // that the UI hasn't added a branch for yet.
    const keeper = {
      metadata: { name: 'future-test', creationTimestamp: '2025-01-01T00:00:00Z' },
      spec: { description: 'Future keeper', futureProvider: { endpoint: 'https://example.com' } },
      status: {},
    } as unknown as Keeper;
    const listItem = keeperToListItem(keeper);

    expect(listItem.type).toBe('unknown');
    expect(listItem.config).toBe('');
  });

  it('handles missing metadata.name gracefully', () => {
    const keeper = {
      metadata: {},
      spec: { description: 'Test', aws: { region: 'us-east-1' } },
      status: {},
    } as Keeper;
    const listItem = keeperToListItem(keeper);

    expect(listItem.name).toBe('');
  });
});
