import { act, screen, waitFor } from '@testing-library/react';
import { useLocalStorage } from 'react-use';
import { render } from 'test/test-utils';

import { OrgRole } from '@grafana/data';
import { contextSrv } from 'app/core/services/context_srv';

import { SavedQueryButtons } from './SavedQueryButtons';
import { mockQuery } from './utils/mocks';

jest.mock('react-use', () => ({
  ...jest.requireActual('react-use'),
  useLocalStorage: jest.fn(),
}));

const mockGetDataSourceSrv = jest.fn();

jest.mock('@grafana/runtime', () => ({
  ...jest.requireActual('@grafana/runtime'),
  getDataSourceSrv: () => {
    return {
      get: mockGetDataSourceSrv,
    };
  },
}));

jest.mock('app/core/hooks/useMediaQueryMinWidth', () => ({
  useMediaQueryMinWidth: jest.fn(() => true),
}));

// Mock contextSrv
jest.mock('app/core/services/context_srv', () => ({
  contextSrv: {
    user: {
      uid: 'user123',
      role: 'Editor',
    },
    isEditor: true,
  },
}));

const mockContextSrv = jest.mocked(contextSrv);

const mockUseLocalStorage = useLocalStorage as jest.Mock;

beforeEach(() => {
  (mockContextSrv.user as any) = { uid: 'user123', role: OrgRole.Editor };
  mockContextSrv.isEditor = true;
  // Default mock: datasource exists
  mockGetDataSourceSrv.mockResolvedValue({});
});

describe('SaveQueryButton', () => {
  it('should render badge for the first time', async () => {
    mockUseLocalStorage.mockReturnValue([true, jest.fn()]);

    render(<SavedQueryButtons query={mockQuery} datasourceFilters={[]} />);
    await waitFor(() => {
      expect(screen.getByText('New!')).toBeInTheDocument();
    });
  });

  it('should render save and replace buttons after the first time', async () => {
    mockUseLocalStorage.mockReturnValue([false, jest.fn()]);
    render(<SavedQueryButtons query={mockQuery} datasourceFilters={[]} />);
    await waitFor(() => {
      expect(screen.getByRole('button', { name: 'Save query' })).toBeInTheDocument();
    });
    expect(screen.getByRole('button', { name: 'Replace with saved query' })).toBeInTheDocument();
  });

  it('should not render save button for viewer but render replace button', async () => {
    mockUseLocalStorage.mockReturnValue([true, jest.fn()]);
    contextSrv.user.orgRole = OrgRole.Viewer;
    contextSrv.isEditor = false;
    render(<SavedQueryButtons query={mockQuery} datasourceFilters={[]} />);

    await waitFor(() => {
      expect(screen.getByText('New!')).toBeInTheDocument();
    });
    expect(screen.queryByRole('button', { name: 'Save query' })).not.toBeInTheDocument();
    expect(screen.getByRole('button', { name: 'Replace with saved query' })).toBeInTheDocument();
  });

  it('should not render buttons if datasource does not exist', async () => {
    mockUseLocalStorage.mockReturnValue([true, jest.fn()]);
    // Mock datasource service to return null (datasource doesn't exist)
    mockGetDataSourceSrv.mockResolvedValue(null);

    await act(async () => {
      render(
        <SavedQueryButtons query={{ ...mockQuery, datasource: { uid: 'does-not-exist' } }} datasourceFilters={[]} />
      );
    });
    expect(screen.queryByRole('button', { name: 'Save query' })).not.toBeInTheDocument();
    expect(screen.queryByRole('button', { name: 'Replace with saved query' })).not.toBeInTheDocument();
  });
});
