import type { FeatureToggles } from '@grafana/data';
import { config } from '@grafana/runtime';

import { getEnricherTypeOptions } from './form';

jest.mock('@grafana/runtime', () => ({
  ...jest.requireActual('@grafana/runtime'),
  config: {
    featureToggles: {},
  },
}));

describe('getEnricherTypeOptions', () => {
  beforeEach(() => {
    config.featureToggles = {} as FeatureToggles;
  });

  it('should return standard enricher types without Assistant Investigations when feature toggle is disabled', () => {
    config.featureToggles.alertingEnrichmentAssistantInvestigations = false;

    const options = getEnricherTypeOptions();

    const optionValues = options.map((option) => option.value);
    const optionLabels = options.map((option) => option.label);

    // Should include all standard types
    expect(optionValues).toContain('assign');
    expect(optionValues).toContain('external');
    expect(optionValues).toContain('dsquery');
    expect(optionValues).toContain('asserts');
    expect(optionValues).toContain('sift');
    expect(optionValues).toContain('explain');

    // Should not include assistant
    expect(optionValues).not.toContain('assistant');
    expect(optionLabels).not.toContain('Assistant Investigations');
    expect(options).toHaveLength(6);
  });

  it('should return standard enricher types without Assistant Investigations when feature toggle is undefined', () => {
    // Feature toggle not set at all
    delete config.featureToggles.alertingEnrichmentAssistantInvestigations;

    const options = getEnricherTypeOptions();

    const optionValues = options.map((option) => option.value);
    expect(optionValues).not.toContain('assistant');
    expect(options).toHaveLength(6);
  });

  it('should include Assistant Investigations when feature toggle is enabled', () => {
    config.featureToggles.alertingEnrichmentAssistantInvestigations = true;

    const options = getEnricherTypeOptions();

    const optionValues = options.map((option) => option.value);
    const optionLabels = options.map((option) => option.label);

    // Should include all standard types
    expect(optionValues).toContain('assign');
    expect(optionValues).toContain('external');
    expect(optionValues).toContain('dsquery');
    expect(optionValues).toContain('asserts');
    expect(optionValues).toContain('sift');
    expect(optionValues).toContain('explain');

    // Should also include assistant
    expect(optionValues).toContain('assistant');
    expect(optionLabels).toContain('Assistant Investigations');
    expect(options).toHaveLength(7);

    const assistantInvestigationsOption = options.find((option) => option.value === 'assistant');
    expect(assistantInvestigationsOption).toEqual({
      description: 'Start Assistant Investigation for alerts',
      label: 'Assistant Investigations',
      value: 'assistant',
    });
  });
});
