import { config, getBackendSrv, isFetchError } from '@grafana/runtime';
import { extractErrorMessage } from 'app/api/utils';
import { contextSrv } from 'app/core/core';
import { AccessControlAction } from 'app/types/accessControl';

import { SCIMFormData, SCIMSettingsData, SCIMUser } from '../../../types';

import { logInfo, logError, logWarning, logMeasurement } from './logger';

// SCIM API constants
const SCIM_API_GROUP = 'scim.grafana.app';
const SCIM_API_VERSION = 'v0alpha1';

// Error messages
const ERROR_MESSAGES = {
  ACCESS_DENIED: 'Access denied',
  CONFIG_FETCH_FAILED: 'Failed to fetch SCIM configuration',
  CONFIG_UPDATE_FAILED: 'Failed to update SCIM configuration',
  CONFIG_RESET_FAILED: 'Failed to reset SCIM configuration',
  USERS_FETCH_FAILED: 'Failed to fetch SCIM users',
} as const;

/**
 * Get the SCIM config endpoint URL with proper namespace
 * @returns string - The SCIM config endpoint URL
 */
function getConfigUrl(): string {
  const namespace = config.namespace;
  return `/apis/scim.grafana.app/v0alpha1/namespaces/${namespace}/config/default`;
}

/**
 * Update SCIM settings with the provided form data
 * @param formData - The SCIM configuration data to update
 * @returns Promise<boolean> - True if successful, throws error if failed
 */
export async function updateSettings(formData: SCIMFormData): Promise<boolean> {
  if (!contextSrv.hasPermission(AccessControlAction.SettingsWrite)) {
    throw new Error(ERROR_MESSAGES.ACCESS_DENIED);
  }

  const startTime = performance.now();

  try {
    const configUrl = getConfigUrl();
    const currentConfig = await getBackendSrv().get(configUrl);

    // Update the config with the new form data
    const updatedConfig = {
      ...currentConfig,
      spec: {
        ...currentConfig.spec,
        enableUserSync: formData.userSyncEnabled,
        enableGroupSync: formData.groupSyncEnabled,
        rejectNonProvisionedUsers: formData.rejectNonProvisionedUsers,
      },
    };

    await getBackendSrv().put(configUrl, updatedConfig);

    logMeasurement(
      'scimConfigUpdate',
      {
        duration: performance.now() - startTime,
        loadTimeMs: performance.now() - startTime,
      },
      {
        method: 'PUT',
        endpoint: 'config',
        userSyncEnabled: String(formData.userSyncEnabled),
        groupSyncEnabled: String(formData.groupSyncEnabled),
      }
    );

    logInfo('SCIM configuration updated successfully', {
      userSyncEnabled: String(formData.userSyncEnabled),
      groupSyncEnabled: String(formData.groupSyncEnabled),
      rejectNonProvisionedUsers: String(formData.rejectNonProvisionedUsers),
    });

    return true;
  } catch (error) {
    const errorMessage = extractErrorMessage(error);
    logError(new Error(ERROR_MESSAGES.CONFIG_UPDATE_FAILED), {
      error: errorMessage,
      userSyncEnabled: String(formData.userSyncEnabled),
      groupSyncEnabled: String(formData.groupSyncEnabled),
      rejectNonProvisionedUsers: String(formData.rejectNonProvisionedUsers),
    });
    throw new Error(`${ERROR_MESSAGES.CONFIG_UPDATE_FAILED}: ${errorMessage}`);
  }
}

/**
 * Fetch SCIM settings from the API, with fallback to static config
 * If dynamic config doesn't exist, attempts to create it with static config values
 * @returns Promise<SCIMSettingsData> - The SCIM settings data
 */
export async function getSettings(): Promise<SCIMSettingsData> {
  if (!contextSrv.hasPermission(AccessControlAction.SettingsRead)) {
    throw new Error(ERROR_MESSAGES.ACCESS_DENIED);
  }

  const startTime = performance.now();

  try {
    // Always check static settings first
    const staticSettings = await getStaticSettings();

    // Try to get dynamic settings
    const dynamicSettings = await getDynamicSettings();

    if (dynamicSettings) {
      // Dynamic config exists, use it but preserve static settings info
      const result = {
        ...dynamicSettings,
        hasStaticSettings: staticSettings.hasStaticSettings,
      };

      logMeasurement(
        'scimConfigFetch',
        {
          duration: performance.now() - startTime,
          loadTimeMs: performance.now() - startTime,
        },
        {
          method: 'GET',
          endpoint: 'config',
          source: 'dynamic',
          hasStaticSettings: String(staticSettings.hasStaticSettings),
        }
      );

      return result;
    }

    // No dynamic config exists, try to create it with static config values
    try {
      await upsertSettings(staticSettings);

      logMeasurement(
        'scimConfigFetch',
        {
          duration: performance.now() - startTime,
          loadTimeMs: performance.now() - startTime,
        },
        {
          method: 'GET',
          endpoint: 'config',
          source: 'static-created',
          hasStaticSettings: String(staticSettings.hasStaticSettings),
        }
      );

      return staticSettings;
    } catch (upsertError) {
      // If upsert fails, fallback to static settings
      const errorMessage = extractErrorMessage(upsertError);
      logWarning('Failed to create dynamic SCIM config, using static settings', {
        error: errorMessage,
        hasStaticSettings: String(staticSettings.hasStaticSettings),
      });

      return staticSettings;
    }
  } catch (error) {
    const errorMessage = extractErrorMessage(error);
    logError(new Error(ERROR_MESSAGES.CONFIG_FETCH_FAILED), { error: errorMessage });
    throw new Error(`${ERROR_MESSAGES.CONFIG_FETCH_FAILED}: ${errorMessage}`);
  }
}

/**
 * Fetch static SCIM settings from configuration file
 * @returns Promise<SCIMSettingsData> - Static SCIM settings from configuration file
 */
export async function getStaticSettings(): Promise<SCIMSettingsData> {
  try {
    // Use verbose settings API to get source information
    const verboseSettings =
      await getBackendSrv().get<Record<string, Record<string, Record<string, string>>>>('/api/admin/settings-verbose');

    // Check if any SCIM settings are actually configured in configuration file
    const scimSection = verboseSettings['auth.scim'];

    if (!scimSection) {
      return {
        userSyncEnabled: false,
        groupSyncEnabled: false,
        rejectNonProvisionedUsers: false,
        hasStaticSettings: false,
      };
    }

    // Check if any SCIM settings are explicitly configured in the configuration file
    const hasStaticSettings = ['user_sync_enabled', 'group_sync_enabled', 'reject_non_provisioned_users'].some(
      (key) => scimSection[key]?.system !== undefined
    );

    const result = {
      userSyncEnabled: getSettingValue(scimSection, 'user_sync_enabled'),
      groupSyncEnabled: getSettingValue(scimSection, 'group_sync_enabled'),
      rejectNonProvisionedUsers: getSettingValue(scimSection, 'reject_non_provisioned_users'),
      hasStaticSettings,
    };

    logInfo('Fetched static SCIM settings', {
      hasStaticSettings: String(hasStaticSettings),
      userSyncEnabled: String(result.userSyncEnabled),
      groupSyncEnabled: String(result.groupSyncEnabled),
      rejectNonProvisionedUsers: String(result.rejectNonProvisionedUsers),
    });

    return result;
  } catch (error) {
    const errorMessage = extractErrorMessage(error);
    logError(new Error('Failed to fetch static SCIM settings'), { error: errorMessage });
    // Re-throw the error instead of returning default values
    throw new Error(`Failed to fetch static SCIM settings: ${errorMessage}`);
  }
}

/**
 * Helper function to get setting value from configuration file
 * @param scimSection - The SCIM section from verbose settings
 * @param key - The setting key to retrieve
 * @returns boolean - The setting value as boolean
 */
const getSettingValue = (scimSection: Record<string, Record<string, string>>, key: string): boolean => {
  const setting = scimSection[key];
  if (!setting) {
    return false;
  }

  return setting.system === 'true';
};

/**
 * Fetch dynamic SCIM settings from the API
 * @returns Promise<SCIMSettingsData | null> - Dynamic settings if available, null if not found or error
 */
export async function getDynamicSettings(): Promise<SCIMSettingsData | null> {
  try {
    const configUrl = getConfigUrl();
    const scimConfig = await getBackendSrv().get(configUrl, undefined, undefined, { showErrorAlert: false });

    const result = {
      userSyncEnabled: scimConfig.spec?.enableUserSync || false,
      groupSyncEnabled: scimConfig.spec?.enableGroupSync || false,
      rejectNonProvisionedUsers: scimConfig.spec?.rejectNonProvisionedUsers || false,
    };

    logInfo('Fetched dynamic SCIM settings', {
      userSyncEnabled: String(result.userSyncEnabled),
      groupSyncEnabled: String(result.groupSyncEnabled),
      rejectNonProvisionedUsers: String(result.rejectNonProvisionedUsers),
    });

    return result;
  } catch (error) {
    // Don't log as error since this is expected when no dynamic config exists
    if (isFetchError(error) && error.status === 404) {
      logInfo('No dynamic SCIM config found, will use static settings');
      return null;
    }

    const errorMessage = extractErrorMessage(error);
    logWarning('Unexpected error when fetching SCIM config', { error: errorMessage });
    return null;
  }
}

/**
 * Upsert dynamic SCIM settings
 * @param staticSettings - The static settings to use as base for dynamic config
 * @returns Promise<any> - The result of the upsert operation
 */
async function upsertSettings(staticSettings: SCIMSettingsData): Promise<any> {
  const configData = {
    apiVersion: `${SCIM_API_GROUP}/${SCIM_API_VERSION}`,
    kind: 'SCIMConfig',
    metadata: {
      name: 'default',
    },
    spec: {
      enableUserSync: staticSettings.userSyncEnabled,
      enableGroupSync: staticSettings.groupSyncEnabled,
      rejectNonProvisionedUsers: staticSettings.rejectNonProvisionedUsers,
    },
  };

  const startTime = performance.now();

  try {
    const configUrl = getConfigUrl();

    // Use PUT to upsert the config (create if doesn't exist, update if it does)
    const result = await getBackendSrv().put(configUrl, configData);

    logMeasurement(
      'scimConfigUpsert',
      {
        duration: performance.now() - startTime,
        loadTimeMs: performance.now() - startTime,
      },
      {
        method: 'PUT',
        endpoint: 'config',
        userSyncEnabled: String(staticSettings.userSyncEnabled),
        groupSyncEnabled: String(staticSettings.groupSyncEnabled),
        rejectNonProvisionedUsers: String(staticSettings.rejectNonProvisionedUsers),
      }
    );

    logInfo('Successfully upserted SCIM config', {
      userSyncEnabled: String(staticSettings.userSyncEnabled),
      groupSyncEnabled: String(staticSettings.groupSyncEnabled),
      rejectNonProvisionedUsers: String(staticSettings.rejectNonProvisionedUsers),
    });

    return result;
  } catch (error) {
    const errorMessage = extractErrorMessage(error);
    logError(new Error('Failed to upsert SCIM config'), {
      error: errorMessage,
      userSyncEnabled: String(staticSettings.userSyncEnabled),
      groupSyncEnabled: String(staticSettings.groupSyncEnabled),
      rejectNonProvisionedUsers: String(staticSettings.rejectNonProvisionedUsers),
    });
    throw new Error(`Failed to upsert SCIM config: ${errorMessage}`);
  }
}

/**
 * Fetch SCIM provisioned users
 * @returns Promise<SCIMUser[]> - Array of SCIM provisioned users
 */
export async function fetchSCIMUsers(): Promise<SCIMUser[]> {
  const startTime = performance.now();

  try {
    const result = await getBackendSrv().get('/api/users/search?perpage=1000&isProvisioned=true');
    const users = result.users || [];

    logMeasurement(
      'scimUsersFetch',
      {
        duration: performance.now() - startTime,
        loadTimeMs: performance.now() - startTime,
      },
      {
        method: 'GET',
        endpoint: 'users',
        userCount: users.length,
      }
    );

    logInfo('Fetched SCIM users', { userCount: users.length });

    return users;
  } catch (error) {
    const errorMessage = extractErrorMessage(error);
    logError(new Error(ERROR_MESSAGES.USERS_FETCH_FAILED), { error: errorMessage });
    throw new Error(`${ERROR_MESSAGES.USERS_FETCH_FAILED}: ${errorMessage}`);
  }
}

/**
 * Reset SCIM configuration by deleting the dynamic config
 * @returns Promise<void>
 */
export async function deleteSettings(): Promise<void> {
  const startTime = performance.now();

  try {
    const configUrl = getConfigUrl();
    await getBackendSrv().delete(configUrl);

    logMeasurement(
      'scimConfigReset',
      {
        duration: performance.now() - startTime,
        loadTimeMs: performance.now() - startTime,
      },
      {
        method: 'DELETE',
        endpoint: 'config',
      }
    );

    logInfo('SCIM configuration reset successfully');
  } catch (error) {
    const errorMessage = extractErrorMessage(error);
    logError(new Error(ERROR_MESSAGES.CONFIG_RESET_FAILED), { error: errorMessage });
    throw new Error(`${ERROR_MESSAGES.CONFIG_RESET_FAILED}: ${errorMessage}`);
  }
}

/**
 * Get domain information for the current Grafana instance
 * @returns Object with domain and subdomain information
 */
export function getDomain() {
  const baseUrl = config.appSubUrl || window.location.origin;
  if (baseUrl.includes('localhost') || baseUrl.includes('127.0.0.1')) {
    return { domain: 'localhost', subdomain: 'local' };
  }

  const url = new URL(baseUrl);
  const hostname = url.hostname;
  const parts = hostname.split('.');

  if (parts.length >= 3 && hostname.endsWith('.grafana.net')) {
    return { domain: hostname, subdomain: parts[0] };
  }

  return { domain: hostname, subdomain: 'unknown' };
}

/**
 * Get the base URL for SCIM operations
 * @returns Promise<string> - The base URL with proper namespace
 */
export async function getBaseUrl(): Promise<string> {
  const baseUrl = config.appSubUrl || window.location.origin;
  const namespace = config.namespace;

  // If namespace is in stacks format, use it directly
  if (namespace.startsWith('stacks-')) {
    return `${baseUrl}/apis/${SCIM_API_GROUP}/${SCIM_API_VERSION}/namespaces/${namespace}`;
  }

  // For default namespace, try to get stack ID from settings
  if (namespace === 'default') {
    try {
      const settings = await getBackendSrv().get('/api/admin/settings');
      if (settings && settings.environment && settings.environment.stack_id) {
        return `${baseUrl}/apis/${SCIM_API_GROUP}/${SCIM_API_VERSION}/namespaces/stacks-${settings.environment.stack_id}`;
      }
    } catch (error) {
      // Fallback to default namespace
    }
  }

  // For other namespaces, use as-is
  return `${baseUrl}/apis/${SCIM_API_GROUP}/${SCIM_API_VERSION}/namespaces/${namespace}`;
}

/**
 * Get stack ID for display purposes
 * @returns string - The stack ID or appropriate fallback message
 */
export function getStackId(): string {
  // Extract stack ID from namespace if it follows the stacks-{id} pattern
  const namespace = config.namespace;
  if (namespace.startsWith('stacks-')) {
    return namespace.replace('stacks-', '');
  }

  // For other namespaces, return the namespace itself
  return namespace;
}
