import { css } from '@emotion/css';

import { GrafanaTheme2 } from '@grafana/data';
import { Trans, t } from '@grafana/i18n';
import { Alert, Box, Button, Stack, Text, useStyles2 } from '@grafana/ui';

import { SCIMSettingsData } from '../../../types';

const getStyles = (theme: GrafanaTheme2) => ({
  section: css({
    marginBottom: theme.spacing(4),
  }),
});

interface Props {
  scimStaticSettingsData: SCIMSettingsData;
  onResetClick: () => void;
  isResetting: boolean;
}

export const SCIMConfigAlert = ({ scimStaticSettingsData, onResetClick, isResetting }: Props) => {
  const styles = useStyles2(getStyles);

  // Only show if there are actual static settings in config.ini
  if (!scimStaticSettingsData.hasStaticSettings) {
    return null;
  }

  return (
    <div className={styles.section}>
      <Alert severity="info" title={t('scim.config.scimSettingsInfo', 'Configuration File Contains SCIM Settings')}>
        <Stack direction="column" gap={2}>
          <Text variant="bodySmall">
            <Trans i18nKey="scim.config.scimSettingsDescription">
              Your configuration file contains SCIM settings. You can update them on this page, or revert to use the
              settings from your configuration file.
            </Trans>
          </Text>
          <Box display="flex" justifyContent="flex-start">
            <Button variant="destructive" size="sm" onClick={onResetClick} icon="trash-alt" disabled={isResetting}>
              {isResetting
                ? t('scim.config.resetting', 'Resetting...')
                : t('scim.config.resetToIniSettings', 'Revert to use configuration file settings')}
            </Button>
          </Box>
        </Stack>
      </Alert>
    </div>
  );
};
