import { css } from '@emotion/css';
import { useEffect, useState } from 'react';

import { GrafanaTheme2 } from '@grafana/data';
import { t } from '@grafana/i18n';
import { Drawer, Icon, Stack, Text, Tooltip, Tab, TabsBar, useStyles2 } from '@grafana/ui';
import { contextSrv } from 'app/core/services/context_srv';
import { SavedQuery } from 'app/features/explore/QueryLibrary/types';

import { useQueryLibraryContext } from '../../features/explore/QueryLibrary/QueryLibraryContext';

import { QueryLibrary } from './QueryLibrary/QueryLibrary';
import { QueryLibraryInteractions } from './QueryLibraryAnalyticsEvents';
import { getUserStorageFavorites, setUserStorageFavorites } from './utils/favorites';

const FEEDBACK_URL =
  'https://docs.google.com/forms/d/e/1FAIpQLScHILHDfIJG8ChaU42AWoGVBR78hIwvvUE-15_A0G5miHgvhQ/viewform?usp=header';

export enum QueryLibraryTab {
  ALL = 'all',
  FAVORITES = 'favorites',
  RECENT = 'history',
  FEEDBACK = 'feedback',
}

type Props = {
  isOpen: boolean;
  // List of datasource names to filter query templates by
  activeDatasources: string[];
  close: () => void;
  // Enhanced drawer options
  highlightedQuery?: string;
  newQuery?: SavedQuery;
};

export const QUERY_LIBRARY_LOCAL_STORAGE_KEYS = {
  explore: {
    newButton: 'grafana.explore.query-library.newButton',
  },
};

/**
 * Drawer with query library feature. Handles its own state and should be included in some top level component.
 */
export function QueryLibraryDrawer({ isOpen, activeDatasources, close, highlightedQuery, newQuery }: Props) {
  const [activeTab, setActiveTab] = useState(QueryLibraryTab.ALL);
  const [userFavorites, setUserFavorites] = useState<{ [key: string]: boolean }>({});
  const { triggerAnalyticsEvent } = useQueryLibraryContext();
  const styles = useStyles2(getStyles);

  useEffect(() => {
    getUserStorageFavorites().then((value) => setUserFavorites(value));
  }, []);

  useEffect(() => {
    // Update tab to All if editing a new query
    if (newQuery && activeTab !== QueryLibraryTab.ALL) {
      setActiveTab(QueryLibraryTab.ALL);
    }
  }, [newQuery, activeTab]);

  const onFavorite = async (uid: string) => {
    const prevFavorites = { ...userFavorites };

    const newUserFavorites = { ...prevFavorites, [uid]: true };
    try {
      setUserFavorites(newUserFavorites);
      await setUserStorageFavorites(newUserFavorites);
      triggerAnalyticsEvent(QueryLibraryInteractions.favoriteQueryClicked, { favoriteState: true });
    } catch (e) {
      setUserFavorites(prevFavorites);
    }
  };

  const onUnfavorite = async (uid: string) => {
    const prevFavorites = { ...userFavorites };

    const newUserFavorites = { ...prevFavorites };
    delete newUserFavorites[uid];
    try {
      setUserFavorites(newUserFavorites);
      await setUserStorageFavorites(newUserFavorites);
      triggerAnalyticsEvent(QueryLibraryInteractions.favoriteQueryClicked, { favoriteState: false });
    } catch (e) {
      setUserFavorites(prevFavorites);
    }
  };

  const handleTabChange = (tab: QueryLibraryTab) => {
    triggerAnalyticsEvent(QueryLibraryInteractions.tabClicked, { tab });
    setActiveTab(tab);
  };

  return (
    isOpen && (
      <Drawer
        title={
          <Stack alignItems="center">
            <Text element="h3">{t('query-library.drawer.title', 'Saved queries')}</Text>
            <Tooltip
              placement="right"
              content={t(`query-library.drawer.tooltip`, 'Right now, each organization can save up to 1000 queries')}
            >
              <Icon name="info-circle" />
            </Tooltip>
          </Stack>
        }
        onClose={close}
        scrollableContent={false}
        tabs={
          <TabsBar>
            <Tab
              label={t('query-library.tabs.all', 'All')}
              active={activeTab === QueryLibraryTab.ALL}
              onChangeTab={() => handleTabChange(QueryLibraryTab.ALL)}
            />
            <Tab
              label={t('query-library.tabs.favorites', 'Favorites')}
              active={activeTab === QueryLibraryTab.FAVORITES}
              onChangeTab={() => handleTabChange(QueryLibraryTab.FAVORITES)}
              disabled={!!newQuery}
            />
            {!contextSrv.hasRole('Viewer') && (
              <Tab
                label={t('query-library.tabs.recent', 'Recent')}
                active={activeTab === QueryLibraryTab.RECENT}
                onChangeTab={() => handleTabChange(QueryLibraryTab.RECENT)}
                disabled={!!newQuery}
              />
            )}
            <Tab
              icon="comment-alt-message"
              href={FEEDBACK_URL}
              target="_blank"
              label={t('query-library.tabs.feedback', 'Give feedback')}
              active={activeTab === QueryLibraryTab.FEEDBACK}
              className={styles.feedbackTab}
            />
          </TabsBar>
        }
      >
        <QueryLibrary
          activeDatasources={activeDatasources}
          activeTab={activeTab}
          userFavorites={userFavorites}
          onFavorite={onFavorite}
          onUnfavorite={onUnfavorite}
          highlightedQuery={highlightedQuery}
          newQuery={newQuery}
        />
      </Drawer>
    )
  );
}

const getStyles = (theme: GrafanaTheme2) => ({
  feedbackTab: css({
    marginLeft: 'auto',
  }),
});
