import { uniqBy } from 'lodash';
import { useEffect, useMemo, useCallback, useState } from 'react';
import { useAsync } from 'react-use';

import { SelectableValue } from '@grafana/data';
import { t } from '@grafana/i18n';
import { Box, Divider, EmptyState, Stack } from '@grafana/ui';
import { useListQueryQuery } from 'app/extensions/api/clients/queries/v1beta1';

import { TermCount } from '../../../core/components/TagFilter/TagFilter';
import { SavedQuery } from '../../../features/explore/QueryLibrary/types';
import { QueryLibraryTab } from '../QueryLibraryDrawer';
import { newestSortingOption } from '../QueryLibrarySortingOptions';
import { selectors } from '../e2e-selectors/selectors';
import { useGetSavedQueries } from '../hooks/useGetSavedQueries';
import { useGetNewSavedQuery } from '../utils/dataFetching';
import { fetchQueryHistory } from '../utils/fetchQueryHistory';
import { convertToMapTagCount } from '../utils/mappers';
import { searchQueryLibrary } from '../utils/search';

import { QueryLibraryContent } from './QueryLibraryContent';
import { QueryLibraryFilters } from './QueryLibraryFilters';

export interface QueryLibraryProps {
  // list of active datasources to filter the query library by
  activeDatasources: string[];
  activeTab: QueryLibraryTab;
  userFavorites: { [key: string]: boolean };
  onFavorite: (uid: string) => void;
  onUnfavorite: (uid: string) => void;
  // Enhanced options (currently unused, reserved for future implementation of overriding query library from explore)
  highlightedQuery?: string;
  // New query to be added to the library. Also used for duplicate query flow.
  newQuery?: SavedQuery;
}

export function QueryLibrary({
  activeDatasources,
  activeTab,
  userFavorites,
  onFavorite,
  onUnfavorite,
  highlightedQuery,
  newQuery: query,
}: QueryLibraryProps) {
  const [searchQuery, setSearchQuery] = useState('');
  const [datasourceFilters, setDatasourceFilters] = useState<Array<SelectableValue<string>>>(
    activeDatasources.map((ds) => ({ value: ds, label: ds }))
  );
  const [userFilters, setUserFilters] = useState<Array<SelectableValue<string>>>([]);
  const [sortingOption, setSortingOption] = useState<SelectableValue | undefined>(newestSortingOption());
  const [tagFilters, setTagFilters] = useState<string[]>([]);

  const {
    data: rawData,
    isLoading: isQueryTemplatesLoading,
    error,
  } = useListQueryQuery({}, { refetchOnMountOrArgChange: true });

  const { value: savedQueries, loading: isSavedQueriesLoading } = useGetSavedQueries(rawData);
  const { data: newQuery, isLoading: isNewQueryLoading, isError: isNewQueryError } = useGetNewSavedQuery(query?.query);

  // Filtering right now is done just on the frontend until there is better backend support for this.
  const filteredRows = useMemo(
    () =>
      savedQueries
        ? searchQueryLibrary(
            savedQueries,
            searchQuery,
            datasourceFilters.map((f) => f.value || ''),
            userFilters.map((f) => f.value || ''),
            tagFilters,
            activeTab,
            userFavorites,
            sortingOption?.sort
          )
        : [],
    [savedQueries, searchQuery, datasourceFilters, userFilters, tagFilters, sortingOption, activeTab, userFavorites]
  );

  const queryHistory = useAsync(async () => {
    return await fetchQueryHistory();
  }, []);

  // Adds the new query datasource to the active datasource filters if it's not already.
  useEffect(() => {
    if (
      newQuery?.datasourceName &&
      !datasourceFilters.some((ds) => ds.value === newQuery.datasourceName) &&
      datasourceFilters.length
    ) {
      setDatasourceFilters([...datasourceFilters, { value: newQuery.datasourceName, label: newQuery.datasourceName }]);
    }
  }, [newQuery, datasourceFilters, setDatasourceFilters]);

  const isFiltered = Boolean(
    searchQuery || datasourceFilters.length > 0 || userFilters.length > 0 || tagFilters.length > 0
  );

  const isLoading =
    isQueryTemplatesLoading ||
    isSavedQueriesLoading ||
    typeof filteredRows === 'undefined' ||
    (isSavedQueriesLoading && !filteredRows.length);

  const users = uniqBy(
    savedQueries?.map(({ user }) => user),
    'uid'
  );
  const datasourceNames = useMemo(() => {
    return savedQueries ? uniqBy(savedQueries, 'datasourceName').map((row) => row.datasourceName) : [];
  }, [savedQueries]);

  const getTagOptions = useCallback(async (): Promise<TermCount[]> => {
    return convertToMapTagCount({ loading: isSavedQueriesLoading, value: savedQueries });
  }, [savedQueries, isSavedQueriesLoading]);

  let libraryRows = filteredRows;
  let usingHistory = false;
  if (activeTab === QueryLibraryTab.RECENT) {
    libraryRows = queryHistory.value || [];
    usingHistory = true;
  }

  const libraryContent = useMemo(() => {
    if ((isLoading && !newQuery) || isNewQueryLoading) {
      return <QueryLibraryContent.Skeleton skeletonDetails />;
    } else {
      return (
        <QueryLibraryContent
          isFiltered={isFiltered}
          usingHistory={usingHistory}
          queryRows={libraryRows || []}
          userFavorites={userFavorites}
          onFavorite={onFavorite}
          onUnfavorite={onUnfavorite}
          highlightedQuery={highlightedQuery}
          newQuery={newQuery}
          activeTab={activeTab}
          isLoading={isLoading}
        />
      );
    }
  }, [
    libraryRows,
    isLoading,
    isFiltered,
    onFavorite,
    onUnfavorite,
    userFavorites,
    highlightedQuery,
    usingHistory,
    newQuery,
    isNewQueryLoading,
    activeTab,
  ]);

  if (error || isNewQueryError) {
    return (
      <EmptyState variant="not-found" message={t('query-library.error-state.title', 'Something went wrong!')}>
        {error instanceof Error ? error.message : ''}
      </EmptyState>
    );
  }

  const showFilters = activeTab !== QueryLibraryTab.RECENT;

  return (
    <Stack data-testid={selectors.components.queryLibraryDrawer.content} height="100%" direction="column" gap={0}>
      {showFilters && (
        <Box backgroundColor="primary" paddingBottom={2}>
          <QueryLibraryFilters
            datasourceFilterOptions={datasourceNames.map((r) => ({
              value: r,
              label: r,
            }))}
            datasourceFilters={datasourceFilters}
            disabled={isLoading || !!newQuery}
            onChangeDatasourceFilters={setDatasourceFilters}
            onChangeSearchQuery={setSearchQuery}
            onChangeSortingOption={setSortingOption}
            onChangeUserFilters={setUserFilters}
            onChangeTagFilters={setTagFilters}
            searchQuery={searchQuery}
            sortingOption={sortingOption}
            userFilterOptions={users?.map((user) => ({
              value: user?.uid,
              label: user?.displayName,
            }))}
            userFilters={userFilters}
            tagFilters={tagFilters}
            getTagOptions={getTagOptions}
          />
        </Box>
      )}
      {showFilters && <Divider spacing={0} />}
      <Stack direction="column" flex={1} minHeight={0}>
        {libraryContent}
      </Stack>
    </Stack>
  );
}
