import { PropsWithChildren, useCallback, useEffect, useMemo, useState } from 'react';

import { CoreApp } from '@grafana/data';
import { config } from '@grafana/runtime';
import { DataQuery } from '@grafana/schema';
import { QueryLibraryContext, QueryLibraryDrawerOptions } from 'app/features/explore/QueryLibrary/QueryLibraryContext';

import { QueryTemplate, type OnSelectQueryType } from '../../features/explore/QueryLibrary/types';

import { QueryLibraryInteractions } from './QueryLibraryAnalyticsEvents';
import { QueryLibraryDrawer } from './QueryLibraryDrawer';
import { SaveQueryButton } from './SaveQueryButton';
import { QueryLibraryEventsPropertyMap } from './types';

import { getQueryLibraryDrawerAction, getQueryLibraryRenderContext } from './index';

export function QueryLibraryContextProvider({ children }: PropsWithChildren) {
  const [isDrawerOpen, setIsDrawerOpen] = useState<boolean>(false);
  const [activeDatasources, setActiveDatasources] = useState<string[]>([]);

  const [newQuery, setNewQuery] = useState<QueryTemplate | undefined>(undefined);
  // need to curry the no-op due to the special setState((prevState) => {}) pattern
  const [onSelectQuery, setOnSelectQuery] = useState<OnSelectQueryType>(() => () => {});
  const [context, setContext] = useState('unknown');
  const [onSave, setOnSave] = useState<(() => void) | undefined>(undefined);

  // Enhanced drawer options
  const [highlightedQuery, setHighlightedQuery] = useState<string | undefined>(undefined);

  // Auto-clear highlight after 3 seconds
  useEffect(() => {
    if (highlightedQuery) {
      const timer = setTimeout(() => {
        setHighlightedQuery(undefined);
      }, 2000);

      return () => clearTimeout(timer);
    }
    return undefined;
  }, [highlightedQuery]);

  const triggerAnalyticsEvent = useCallback(
    (
      handleAnalyticEvent: (properties?: QueryLibraryEventsPropertyMap) => void,
      properties?: QueryLibraryEventsPropertyMap,
      contextOverride?: string
    ) => {
      const appContext = contextOverride || context;
      const propertiesWithContext = { app: appContext, ...properties };
      handleAnalyticEvent(propertiesWithContext);
    },
    [context]
  );

  const openDrawer = useCallback(
    ({ datasourceFilters, onSelectQuery, options, query }: QueryLibraryDrawerOptions) => {
      setActiveDatasources(datasourceFilters || []);
      setOnSave(() => options?.onSave);
      // need to curry our function due to the special setState((prevState) => {}) pattern
      if (onSelectQuery) {
        setOnSelectQuery(() => onSelectQuery);
      }
      setIsDrawerOpen(true);
      setContext(getQueryLibraryRenderContext(options?.context));
      // Set enhanced options
      setHighlightedQuery(options?.highlightQuery);

      triggerAnalyticsEvent(
        QueryLibraryInteractions.queryLibraryOpened,
        {
          mode: getQueryLibraryDrawerAction({ query, options }),
        },
        options?.context
      );

      if (!!query) {
        setNewQuery({ query });
        triggerAnalyticsEvent(QueryLibraryInteractions.saveQueryToLibraryClicked, undefined, options?.context);
      } else {
        options?.isReplacingQuery
          ? triggerAnalyticsEvent(
              QueryLibraryInteractions.replaceWithQueryFromLibraryClicked,
              undefined,
              options?.context
            )
          : triggerAnalyticsEvent(QueryLibraryInteractions.addQueryFromLibraryClicked, undefined, options?.context);
      }
    },
    [triggerAnalyticsEvent]
  );

  const closeDrawer = useCallback(
    (isSelectingQuery?: boolean, isEditingQuery?: boolean) => {
      setActiveDatasources([]);
      // need to curry the no-op due to the special setState((prevState) => {}) pattern
      setOnSelectQuery(() => () => {});
      setNewQuery(undefined);
      setIsDrawerOpen(false);

      // Clear enhanced options
      setHighlightedQuery(undefined);

      if (isEditingQuery) {
        triggerAnalyticsEvent(QueryLibraryInteractions.queryLibraryClosedToEditQueryInExplore);
      } else if (!isSelectingQuery && !newQuery) {
        triggerAnalyticsEvent(QueryLibraryInteractions.queryLibraryClosedWithoutSelection);
      } else if (newQuery) {
        triggerAnalyticsEvent(QueryLibraryInteractions.queryLibraryClosedWithoutSavingNewQUery);
      }
    },
    [newQuery, triggerAnalyticsEvent]
  );

  const contextVal = useMemo(
    () => ({
      isDrawerOpen,
      openDrawer,
      closeDrawer,
      triggerAnalyticsEvent,
      renderSaveQueryButton: (
        query: DataQuery,
        app?: CoreApp,
        queryLibraryRef?: string,
        onUpdateSuccess?: () => void,
        onSelectQuery?: (query: DataQuery) => void
      ) => (
        <SaveQueryButton
          query={query}
          app={app}
          queryLibraryRef={queryLibraryRef}
          onUpdateSuccess={onUpdateSuccess}
          onSelectQuery={onSelectQuery}
        />
      ),
      queryLibraryEnabled: Boolean(config.featureToggles.queryLibrary),
      context,
      setNewQuery,
      onSave,
    }),
    [isDrawerOpen, openDrawer, closeDrawer, context, setNewQuery, onSave, triggerAnalyticsEvent]
  );

  return (
    <QueryLibraryContext.Provider value={contextVal}>
      {children}
      <QueryLibraryDrawer
        isOpen={isDrawerOpen}
        close={() => closeDrawer(false)}
        activeDatasources={activeDatasources}
        highlightedQuery={highlightedQuery}
        onSelectQuery={(query) => {
          onSelectQuery(query);
          closeDrawer(true);
          triggerAnalyticsEvent(QueryLibraryInteractions.selectQueryClicked);
        }}
        newQuery={newQuery}
      />
    </QueryLibraryContext.Provider>
  );
}
