import { screen } from '@testing-library/react';
import { FormProvider, useForm } from 'react-hook-form';
import { render } from 'test/test-utils';

import { mockQueryTemplateRow } from '../utils/mocks';

import { QueryLibraryActions, QueryLibraryActionsProps } from './QueryLibraryActions';
import { QueryDetails } from './QueryLibraryDetails';

const mockSetNewQuery = jest.fn();
jest.mock('app/features/explore/QueryLibrary/QueryLibraryContext', () => ({
  useQueryLibraryContext: () => ({
    triggerAnalyticsEvent: jest.fn(),
    setNewQuery: mockSetNewQuery,
  }),
}));

describe('QueryLibraryActions', () => {
  const mockOnSelectQuery = jest.fn();
  const defaultProps: QueryLibraryActionsProps = {
    onSelectQuery: mockOnSelectQuery,
    selectedQueryRow: mockQueryTemplateRow,
    isEditingQuery: false,
    onEditQuerySuccess: jest.fn(),
    setIsEditingQuery: jest.fn(),
    isSavingLoading: false,
  };

  const lockedQueryTemplateRow = {
    ...mockQueryTemplateRow,
    isLocked: true,
  };

  const lockedQueryProps: QueryLibraryActionsProps = {
    onSelectQuery: mockOnSelectQuery,
    selectedQueryRow: lockedQueryTemplateRow,
    isEditingQuery: false,
    onEditQuerySuccess: jest.fn(),
    setIsEditingQuery: jest.fn(),
    isSavingLoading: false,
  };

  beforeEach(() => {
    jest.clearAllMocks();
  });

  const QueryLibraryActionsWrapper = (props: QueryLibraryActionsProps) => {
    const methods = useForm<QueryDetails>();

    return (
      <FormProvider {...methods}>
        <QueryLibraryActions {...props} />
      </FormProvider>
    );
  };

  it('should render all action buttons not related to editing', () => {
    render(<QueryLibraryActionsWrapper {...defaultProps} />);
    expect(screen.getByRole('button', { name: 'Duplicate query' })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: 'Lock query' })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: 'Delete query' })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: 'Select query' })).toBeInTheDocument();
    expect(screen.queryByRole('button', { name: 'Save' })).not.toBeInTheDocument();
    expect(screen.queryByRole('button', { name: 'Cancel' })).not.toBeInTheDocument();
  });

  it('should not render update and select actions when query is new', () => {
    render(
      <QueryLibraryActionsWrapper {...defaultProps} selectedQueryRow={{ ...mockQueryTemplateRow, uid: undefined }} />
    );
    expect(screen.queryByRole('button', { name: 'Duplicate query' })).not.toBeInTheDocument();
    expect(screen.queryByRole('button', { name: 'Lock query' })).not.toBeInTheDocument();
    expect(screen.queryByRole('button', { name: 'Delete query' })).not.toBeInTheDocument();
    expect(screen.queryByRole('button', { name: 'Select query' })).not.toBeInTheDocument();

    expect(screen.getByRole('button', { name: 'Save' })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: 'Cancel' })).toBeInTheDocument();
  });

  it('should render editing buttons when is editing query', () => {
    render(<QueryLibraryActionsWrapper {...defaultProps} isEditingQuery={true} />);
    expect(screen.getByRole('button', { name: 'Duplicate query' })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: 'Lock query' })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: 'Delete query' })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: 'Lock query' })).toBeDisabled();

    expect(screen.queryByRole('button', { name: 'Select query' })).not.toBeInTheDocument();
    expect(screen.getByRole('button', { name: 'Save' })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: 'Cancel' })).toBeInTheDocument();
  });

  it('should render unlock button when query is locked', () => {
    render(<QueryLibraryActionsWrapper {...lockedQueryProps} />);
    expect(screen.getByRole('button', { name: 'Unlock query' })).toBeInTheDocument();
  });

  it('should disable lock button when query is being edited', () => {
    render(<QueryLibraryActionsWrapper {...defaultProps} isEditingQuery={true} />);
    expect(screen.getByRole('button', { name: 'Lock query' })).toBeDisabled();
  });

  it('should call onSelectQuery when the select button is clicked', async () => {
    const { user } = render(<QueryLibraryActionsWrapper {...defaultProps} />);
    await user.click(screen.getByRole('button', { name: 'Select query' }));
    expect(mockOnSelectQuery).toHaveBeenCalledWith(mockQueryTemplateRow.query);
  });

  it('should set new query when duplicating query', async () => {
    const { user } = render(<QueryLibraryActionsWrapper {...defaultProps} />);
    await user.click(screen.getByRole('button', { name: 'Duplicate query' }));
    expect(mockSetNewQuery).toHaveBeenCalledWith({
      ...mockQueryTemplateRow,
      uid: undefined,
      title: `${mockQueryTemplateRow.title} Copy`,
    });
  });
});
