import { useLocalStorage } from 'react-use';

import { CoreApp } from '@grafana/data';
import { t } from '@grafana/i18n';
import { DataQuery } from '@grafana/schema';
import { Badge } from '@grafana/ui';
import { QueryOperationAction } from 'app/core/components/QueryOperationRow/QueryOperationAction';

import { QUERY_LIBRARY_LOCAL_STORAGE_KEYS } from './QueryLibraryDrawer';
import { selectors } from './e2e-selectors/selectors';
import { useQueryLibrarySave } from './hooks/useQueryLibrarySave';

interface Props {
  query: DataQuery;
  queryLibraryRef?: string;
  app?: CoreApp;
  onSelectQuery?: (query: DataQuery) => void;
  onUpdateSuccess?: () => void;
}

export function SaveQueryButton({ query, app, queryLibraryRef, onUpdateSuccess, onSelectQuery }: Props) {
  const { saveNewQuery, updateQuery } = useQueryLibrarySave();

  const [showQueryLibraryBadgeButton, setShowQueryLibraryBadgeButton] = useLocalStorage(
    QUERY_LIBRARY_LOCAL_STORAGE_KEYS.explore.newButton,
    true
  );

  return showQueryLibraryBadgeButton ? (
    <Badge
      data-testid={selectors.components.saveQueryButton.button}
      text={t('query-operation.header.save-to-query-library-new', 'New: Save to query library')}
      icon="save"
      color="blue"
      onClick={() => {
        saveNewQuery(query, onSelectQuery, { context: app });
        setShowQueryLibraryBadgeButton(false);
      }}
      style={{ cursor: 'pointer' }}
    />
  ) : !queryLibraryRef ? (
    <QueryOperationAction
      dataTestId={selectors.components.saveQueryButton.button}
      title={t('query-operation.header.save-to-query-library', 'Save to query library')}
      icon="save"
      onClick={() => {
        saveNewQuery(query, onSelectQuery, { context: app });
      }}
      isHighlighted
    />
  ) : (
    <QueryOperationAction
      dataTestId={selectors.components.saveQueryButton.button}
      title={t('query-operation.header.update-query-in-library', 'Update query in library')}
      icon="save"
      onClick={() => {
        updateQuery(query, { queryLibraryRef, context: app }, onUpdateSuccess, onSelectQuery);
      }}
    />
  );
}
