import { Fragment, useEffect, useRef, useState } from 'react';
import { FormProvider, useForm } from 'react-hook-form';

import { Trans, t } from '@grafana/i18n';
import { Box, Divider, EmptyState, ScrollContainer, Stack } from '@grafana/ui';
import { attachSkeleton, SkeletonComponent } from '@grafana/ui/unstable';
import { generatedAPI } from 'app/extensions/api/clients/querylibrary/v0alpha1/endpoints.gen';
import { useQueryLibraryContext } from 'app/features/explore/QueryLibrary/QueryLibraryContext';

import { OnSelectQueryType } from '../../../features/explore/QueryLibrary/types';
import { QueryTemplateRow } from '../types';

import { QueryDetails, QueryLibraryDetails } from './QueryLibraryDetails';
import { QueryLibraryForm } from './QueryLibraryForm';
import { QueryLibraryItem } from './QueryLibraryItem';

export interface QueryLibraryContentProps {
  isFiltered?: boolean;
  onSelectQuery: OnSelectQueryType;
  queryRows: QueryTemplateRow[];
  userFavorites: { [key: string]: boolean };
  onFavorite: (uid: string) => void;
  onUnfavorite: (uid: string) => void;
  highlightedQuery?: string;
  newQuery?: QueryTemplateRow;
}

function QueryLibraryContentComponent({
  isFiltered,
  onSelectQuery,
  queryRows,
  userFavorites,
  onFavorite,
  onUnfavorite,
  highlightedQuery,
  newQuery,
}: QueryLibraryContentProps) {
  const { setNewQuery } = useQueryLibraryContext();

  const [isEditingQuery, setIsEditingQuery] = useState(!!newQuery);
  const [selectedQueryRowIndex, setSelectedQueryRowIndex] = useState<number>(0);
  const queryItemsRef = useRef<HTMLDivElement>(null); // to handle auto-scrolling when highlightedQuery is set
  const scrollContainerRef = useRef<HTMLDivElement>(null);
  const [lastUpdatedUid, setLastUpdatedUid] = useState<string>();

  const isEmpty = queryRows.length === 0 && !newQuery;

  const { isSuccess, isFetching } = generatedAPI.endpoints.listQueryTemplate.useQueryState({});
  const currentQuery: QueryTemplateRow | undefined = newQuery ?? queryRows[selectedQueryRowIndex];

  const { reset, watch, ...rest } = useForm<QueryDetails>({
    defaultValues: {
      title: currentQuery?.title ?? '',
      description: currentQuery?.description ?? '',
      tags: currentQuery?.tags ?? [],
      isVisible: currentQuery?.isVisible ?? true,
    },
  });

  // Resets React Form Hook with new default values when selecting new query in library
  useEffect(() => {
    reset({
      title: currentQuery?.title,
      description: currentQuery?.description,
      tags: currentQuery?.tags ?? [],
      isVisible: currentQuery?.isVisible,
    });
  }, [currentQuery, reset]);

  useEffect(() => {
    if (isSuccess && !isFetching && !newQuery) {
      if (highlightedQuery && queryRows.length > 0) {
        // Auto-select highlighted query when specified
        const highlightIndex = queryRows.findIndex((row) => row.uid === highlightedQuery);
        if (highlightIndex !== -1) {
          setSelectedQueryRowIndex(highlightIndex);
          setIsEditingQuery(false); // Ensure we're not in edit mode

          // Scroll to the highlighted query
          const highlightedElement = queryItemsRef.current?.querySelector(`[data-query-uid="${highlightedQuery}"]`);
          if (highlightedElement) {
            highlightedElement.scrollIntoView({
              behavior: 'smooth',
              block: 'center',
              inline: 'nearest',
            });
          }
        }
      } else {
        const newIndex = queryRows.findIndex((q) => q.uid === lastUpdatedUid);
        if (newIndex !== -1) {
          setSelectedQueryRowIndex(newIndex);
        } else {
          setSelectedQueryRowIndex((prevState) =>
            queryRows[prevState] || prevState === 0 ? prevState : prevState - 1
          );
        }
      }
    }
  }, [queryRows, isSuccess, isFetching, lastUpdatedUid, highlightedQuery, newQuery]);

  useEffect(() => {
    if (newQuery) {
      // Scroll to the new query
      scrollContainerRef.current?.scrollTo({
        top: 0,
        behavior: 'smooth',
      });
    }
  }, [newQuery]);

  const onEditQuerySuccess = (uid: string, isNew?: boolean) => {
    setIsEditingQuery(false);
    if (isNew) {
      setNewQuery(undefined);
    }
    setLastUpdatedUid(uid);
  };

  if (isEmpty) {
    // search miss
    return isFiltered ? (
      <EmptyState message={t('query-library.not-found.title', 'No results found')} variant="not-found">
        <Trans i18nKey="query-library.not-found.message">Try adjusting your search or filter criteria</Trans>
      </EmptyState>
    ) : (
      // true empty state
      <EmptyState
        message={t('query-library.empty-state.title', "You haven't saved any queries to the library yet")}
        variant="call-to-action"
      >
        <Trans i18nKey="query-library.empty-state.message">
          Start adding them from Explore or when editing a dashboard
        </Trans>
      </EmptyState>
    );
  }

  return (
    <FormProvider {...{ reset, watch, ...rest }}>
      <Stack flex={1} gap={0} minHeight={0}>
        <Box display="flex" flex={1} minWidth={0}>
          <ScrollContainer ref={scrollContainerRef}>
            <Stack direction="column" gap={0} flex={1} minWidth={0} role="radiogroup" ref={queryItemsRef}>
              {newQuery && (
                <Stack data-query-uid="new-query">
                  <QueryLibraryItem
                    isSelected
                    onSelectQueryRow={() => {}}
                    queryRow={{ ...currentQuery, title: watch('title') }}
                    isNew
                  />
                </Stack>
              )}
              {queryRows.map((queryRow, index) => (
                <Fragment key={queryRow.uid}>
                  <QueryLibraryItem
                    data-query-uid={queryRow.uid}
                    isSelected={currentQuery?.uid === queryRow.uid}
                    isFavorite={userFavorites?.[queryRow.uid ?? ''] ?? false}
                    onFavorite={() => onFavorite(queryRow.uid ?? '')}
                    onUnfavorite={() => onUnfavorite(queryRow.uid ?? '')}
                    isHighlighted={highlightedQuery === queryRow.uid}
                    onSelectQueryRow={() => {
                      setSelectedQueryRowIndex(index);
                      setIsEditingQuery(false);
                    }}
                    queryRow={{
                      ...queryRow,
                      title: currentQuery?.uid === queryRow.uid && isEditingQuery ? watch('title') : queryRow.title,
                    }}
                    disabled={!!newQuery}
                  />
                  <Divider spacing={0} />
                </Fragment>
              ))}
            </Stack>
          </ScrollContainer>
        </Box>
        <Divider direction="vertical" spacing={0} />
        <Box display="flex" flex={2} minWidth={0}>
          <ScrollContainer>
            <Box
              direction="column"
              display="flex"
              flex={1}
              paddingBottom={0}
              paddingLeft={2}
              paddingRight={1}
              paddingTop={2}
            >
              <QueryLibraryForm
                onSelectQuery={onSelectQuery}
                selectedQueryRow={currentQuery}
                isEditingQuery={isEditingQuery}
                setIsEditingQuery={setIsEditingQuery}
                onEditQuerySuccess={onEditQuerySuccess}
              />
            </Box>
          </ScrollContainer>
        </Box>
      </Stack>
    </FormProvider>
  );
}

const QueryLibraryContentSkeleton: SkeletonComponent = ({ rootProps }) => {
  return (
    <Stack flex={1} gap={0} minHeight={0} {...rootProps}>
      <Box display="flex" flex={1} minWidth={0}>
        <Stack direction="column" flex={1} gap={0} minWidth={0}>
          {new Array(5).fill(0).map((_, index) => (
            <Fragment key={index}>
              <QueryLibraryItem.Skeleton />
              <Divider spacing={0} />
            </Fragment>
          ))}
        </Stack>
      </Box>
      <Divider direction="vertical" spacing={0} />
      <Box display="flex" flex={2} minWidth={0}>
        <Box
          direction="column"
          display="flex"
          flex={1}
          paddingBottom={0}
          paddingLeft={2}
          paddingRight={1}
          paddingTop={2}
        >
          <QueryLibraryDetails.Skeleton />
        </Box>
      </Box>
    </Stack>
  );
};

export const QueryLibraryContent = attachSkeleton(QueryLibraryContentComponent, QueryLibraryContentSkeleton);
