import { screen, waitFor } from '@testing-library/react';

import { isLLMPluginEnabled } from 'app/features/dashboard/components/GenAI/utils';
import { addExtraMiddleware, addRootReducer } from 'app/store/configureStore';

import { render } from '../../../../test/test-utils';
import { alertEnrichmentAPIv0alpha1 } from '../../api/clients/alertenrichment/v0alpha1';

import NewEnrichment from './NewEnrichment';
import { setupEnrichmentMockServer } from './__mocks__/enrichmentApi';

// Mock dependencies
jest.mock('app/features/dashboard/components/GenAI/utils');

// Mock the AlertEnrichmentForm component
jest.mock('./form/AlertEnrichmentForm', () => ({
  AlertEnrichmentForm: jest.fn(({ onSubmit, onCancel, isLoading, llmEnabled }) => (
    <div data-testid="alert-enrichment-form">
      <button
        data-testid="submit-btn"
        onClick={() =>
          onSubmit({
            spec: {
              title: 'Test Enrichment',
              description: 'Test Description',
              steps: [
                {
                  timeout: '30s',
                  type: 'enricher',
                  enricher: {
                    type: 'assign',
                    assign: { annotations: [] },
                  },
                },
              ],
            },
            labelMatchers: [],
            annotationMatchers: [],
          })
        }
      >
        Submit
      </button>
      <button data-testid="cancel-btn" onClick={onCancel}>
        Cancel
      </button>
      <div data-testid="loading-state">{isLoading ? 'loading' : 'not-loading'}</div>
      <div data-testid="llm-state">{llmEnabled ? 'llm-enabled' : 'llm-disabled'}</div>
    </div>
  )),
}));

// Setup MSW for API mocking
setupEnrichmentMockServer();

// Add the alert enrichment API to the store configuration once
beforeAll(() => {
  addRootReducer({
    [alertEnrichmentAPIv0alpha1.reducerPath]: alertEnrichmentAPIv0alpha1.reducer,
  });
  addExtraMiddleware(alertEnrichmentAPIv0alpha1.middleware);
});

describe('NewEnrichment', () => {
  beforeEach(() => {
    jest.clearAllMocks();
    (isLLMPluginEnabled as jest.Mock).mockResolvedValue(true);
  });

  it('renders the form with correct props', async () => {
    render(<NewEnrichment />, {
      historyOptions: { initialEntries: ['/alerting/enrichments/new'] },
    });

    await waitFor(() => {
      expect(screen.getByTestId('alert-enrichment-form')).toBeInTheDocument();
      expect(screen.getByTestId('llm-state')).toHaveTextContent('llm-enabled');
      expect(screen.getByTestId('loading-state')).toHaveTextContent('not-loading');
    });
  });

  it('handles LLM disabled state', async () => {
    (isLLMPluginEnabled as jest.Mock).mockResolvedValue(false);

    render(<NewEnrichment />, {
      historyOptions: { initialEntries: ['/alerting/enrichments/new'] },
    });

    await waitFor(() => {
      expect(screen.getByTestId('llm-state')).toHaveTextContent('llm-disabled');
    });
  });

  it('calls onCancel when cancel button is clicked', async () => {
    const { user } = render(<NewEnrichment />, {
      historyOptions: { initialEntries: ['/alerting/enrichments/new'] },
    });

    await user.click(screen.getByTestId('cancel-btn'));

    // Since we mocked the form component, we can only test that the click handler works
    // The actual navigation is tested in integration tests
    expect(screen.getByTestId('cancel-btn')).toBeInTheDocument();
  });

  it('submits form data correctly', async () => {
    const { user } = render(<NewEnrichment />, {
      historyOptions: { initialEntries: ['/alerting/enrichments/new'] },
    });

    await user.click(screen.getByTestId('submit-btn'));

    // Test that form submission works without errors
    // The actual API call and navigation are handled by the real component
    expect(screen.getByTestId('submit-btn')).toBeInTheDocument();
  });

  it('passes correct props to form component', async () => {
    render(<NewEnrichment />, {
      historyOptions: { initialEntries: ['/alerting/enrichments/new'] },
    });

    await waitFor(() => {
      expect(screen.getByTestId('loading-state')).toHaveTextContent('not-loading');
      expect(screen.getByTestId('llm-state')).toHaveTextContent('llm-enabled');
    });
  });
});
