import { css } from '@emotion/css';
import { useFormContext } from 'react-hook-form';

import { Trans, t } from '@grafana/i18n';
import { getAppEvents } from '@grafana/runtime';
import { Button, IconButton, Stack, Spinner, useStyles2 } from '@grafana/ui';
import { useQueryLibraryContext } from 'app/features/explore/QueryLibrary/QueryLibraryContext';

import { showDiscardAddQueryModal } from '..';
import { notifyApp } from '../../../core/actions';
import { createSuccessNotification } from '../../../core/copy/appNotification';
import { OnSelectQueryType } from '../../../features/explore/QueryLibrary/types';
import { dispatch } from '../../../store/store';
import { ShowConfirmModalEvent } from '../../../types/events';
import {
  useDeleteQueryTemplateMutation,
  useUpdateQueryTemplateMutation,
} from '../../api/clients/querylibrary/v0alpha1';
import { QueryLibraryInteractions } from '../QueryLibraryAnalyticsEvents';
import { selectors } from '../e2e-selectors/selectors';
import { QueryTemplateRow } from '../types';
import { canEditQuery } from '../utils/identity';

import { QueryDetails } from './QueryLibraryDetails';

export interface QueryLibraryActionsProps {
  onSelectQuery: OnSelectQueryType;
  selectedQueryRow: QueryTemplateRow;
  isEditingQuery: boolean;
  setIsEditingQuery: (isEditingQuery: boolean) => void;
  onEditQuerySuccess: (uid: string, isNew?: boolean) => void;
  isSavingLoading: boolean;
}

export function QueryLibraryActions({
  onSelectQuery,
  selectedQueryRow,
  isEditingQuery,
  setIsEditingQuery,
  onEditQuerySuccess,
  isSavingLoading,
}: QueryLibraryActionsProps) {
  const styles = useStyles2(getStyles);
  const { setNewQuery } = useQueryLibraryContext();

  const [deleteQueryTemplate] = useDeleteQueryTemplateMutation();
  const [editQueryTemplate, { isLoading: isUpdateLoading }] = useUpdateQueryTemplateMutation();
  const { triggerAnalyticsEvent } = useQueryLibraryContext();

  const { isLocked } = selectedQueryRow;

  const {
    reset,
    formState: { isDirty },
  } = useFormContext<QueryDetails>();

  const onDeleteQuery = (queryUid: string) => {
    const performDelete = async (queryUid: string) => {
      QueryLibraryInteractions.deleteQueryClicked();
      await deleteQueryTemplate({
        name: queryUid,
      }).unwrap();
      dispatch(notifyApp(createSuccessNotification(t('query-library.notifications.query-deleted', 'Query deleted'))));
      triggerAnalyticsEvent(QueryLibraryInteractions.deleteQueryClicked);
    };

    getAppEvents().publish(
      new ShowConfirmModalEvent({
        title: t('query-library.delete-modal.title', 'Delete query'),
        text: t(
          'query-library.delete-modal.body-text',
          "You're about to remove this query from the query library. This action cannot be undone. Do you want to continue?"
        ),
        yesText: t('query-library.delete-modal.confirm-button', 'Delete query'),
        icon: 'trash-alt',
        onConfirm: () => performDelete(queryUid),
      })
    );
  };

  const onDuplicateQuery = () => {
    triggerAnalyticsEvent(QueryLibraryInteractions.duplicateQueryClicked);
    setNewQuery({
      ...selectedQueryRow,
      uid: undefined,
      title: `${selectedQueryRow.title} ${t('query-library.actions.duplicate-query-title-copy', 'Copy')}`,
    });
    setIsEditingQuery(true);
  };

  const onLockToggle = async () => {
    if (!selectedQueryRow.uid) {
      return;
    }
    triggerAnalyticsEvent(QueryLibraryInteractions.lockQueryClicked, {
      isLocked: !isLocked,
    });
    await editQueryTemplate({
      name: selectedQueryRow.uid || '',
      patch: {
        spec: {
          isLocked: !isLocked,
        },
      },
    }).unwrap();

    onEditQuerySuccess(selectedQueryRow.uid);
  };

  const resetForm = () => {
    reset();
    if (!selectedQueryRow.uid) {
      QueryLibraryInteractions.cancelSaveNewQueryClicked();
      setNewQuery(undefined);
    } else {
      QueryLibraryInteractions.cancelEditClicked();
    }
    setIsEditingQuery(false);
  };

  const onCancelEditClick = () => {
    if (isDirty) {
      showDiscardAddQueryModal(resetForm);
    } else {
      resetForm();
    }
  };

  return (
    <Stack wrap="wrap" justifyContent="space-between">
      {selectedQueryRow.uid && (
        <Stack wrap="wrap" justifyContent="flex-start" alignItems="center">
          <IconButton
            data-testid={selectors.components.queryLibraryDrawer.delete}
            variant="secondary"
            name="trash-alt"
            onClick={() => onDeleteQuery(selectedQueryRow.uid ?? '')}
            tooltip={t('query-library.actions.delete-query-button', 'Delete query')}
            disabled={selectedQueryRow.isLocked || !canEditQuery(selectedQueryRow)}
          />
          <IconButton
            data-testid={selectors.components.queryLibraryDrawer.lock}
            variant={isLocked ? 'primary' : 'secondary'}
            name={isLocked ? 'lock' : 'unlock'}
            onClick={onLockToggle}
            tooltip={
              isLocked
                ? t('query-library.actions.unlock-query-button', 'Unlock query')
                : t('query-library.actions.lock-query-button', 'Lock query')
            }
            disabled={!canEditQuery(selectedQueryRow) || isEditingQuery}
          />
          <IconButton
            data-testid={selectors.components.queryLibraryDrawer.duplicate}
            variant="secondary"
            name="copy"
            onClick={onDuplicateQuery}
            tooltip={t('query-library.actions.duplicate-query-button', 'Duplicate query')}
          />
          {isUpdateLoading && <Spinner />}
        </Stack>
      )}
      {isEditingQuery || !selectedQueryRow.uid ? (
        <div className={styles.editActionsContainer}>
          <Stack alignItems="center">
            <Button variant="secondary" onClick={onCancelEditClick}>
              <Trans i18nKey="explore.query-library.cancel">Cancel</Trans>
            </Button>
            <Button
              variant="primary"
              type="submit"
              disabled={(!isDirty && !!selectedQueryRow.uid) || isSavingLoading}
              icon={isSavingLoading ? 'spinner' : undefined}
              data-testid={selectors.components.queryLibraryDrawer.saveQueryButton}
            >
              {isSavingLoading
                ? t('explore.query-library.saving', 'Saving...')
                : t('explore.query-library.save', 'Save')}
            </Button>
          </Stack>
        </div>
      ) : (
        <Button
          data-testid={selectors.components.queryLibraryDrawer.confirm}
          onClick={() => onSelectQuery(selectedQueryRow.query)}
        >
          <Trans i18nKey="query-library.actions.select-query-button">Select query</Trans>
        </Button>
      )}
    </Stack>
  );
}

const getStyles = () => {
  return {
    editActionsContainer: css({
      marginLeft: 'auto',
    }),
  };
};
