import { jsxs, jsx } from 'react/jsx-runtime';
import { cx, css } from '@emotion/css';
import { useMemo } from 'react';
import { t } from '@grafana/i18n';
import { useStyles2 } from '../../themes/ThemeContext.mjs';
import { getTagColorsFromName } from '../../utils/tags.mjs';
import { IconButton } from '../IconButton/IconButton.mjs';

"use strict";
const TagItem = ({ name, disabled, onRemove, autoColors = true }) => {
  const styles = useStyles2(getStyles);
  const tagStyle = useMemo(() => {
    if (autoColors) {
      const { color, borderColor } = getTagColorsFromName(name);
      return { backgroundColor: color, borderColor };
    }
    return void 0;
  }, [name, autoColors]);
  return /* @__PURE__ */ jsxs("li", { className: cx(styles.itemStyle, !tagStyle && styles.defaultTagColor), style: tagStyle, children: [
    /* @__PURE__ */ jsx("span", { className: styles.nameStyle, children: name }),
    /* @__PURE__ */ jsx(
      IconButton,
      {
        name: "times",
        size: "lg",
        disabled,
        tooltip: t("grafana-ui.tags-input.remove", "Remove tag: {{name}}", { name }),
        onClick: () => onRemove(name),
        className: styles.buttonStyles
      }
    )
  ] });
};
const getStyles = (theme) => {
  const height = theme.spacing.gridSize * 3;
  return {
    itemStyle: css({
      display: "flex",
      gap: "3px",
      alignItems: "center",
      height: `${height}px`,
      lineHeight: `${height - 2}px`,
      borderWidth: "1px",
      borderStyle: "solid",
      borderRadius: theme.shape.radius.default,
      padding: `0 ${theme.spacing(0.5)}`,
      whiteSpace: "nowrap",
      textShadow: "none",
      fontWeight: 500,
      fontSize: theme.typography.size.sm,
      color: "#fff"
    }),
    defaultTagColor: css({
      backgroundColor: theme.colors.background.secondary,
      borderColor: theme.components.input.borderColor,
      color: theme.colors.text.primary
    }),
    nameStyle: css({
      maxWidth: "25ch",
      textOverflow: "ellipsis",
      overflow: "hidden"
    }),
    buttonStyles: css({
      margin: 0,
      "&:hover::before": {
        display: "none"
      }
    })
  };
};

export { TagItem };
//# sourceMappingURL=TagItem.mjs.map
