"use strict";
const ErrorId = 0;
const warningTypes = {
  SubqueryExpr: "This subquery may return only one data point, preventing rate/increase/delta calculations. Use a range at least twice the step size (e.g., [2x:x])."
};
var NodeType = /* @__PURE__ */ ((NodeType2) => {
  NodeType2["SubqueryExpr"] = "SubqueryExpr";
  NodeType2["Duration"] = "NumberDurationLiteralInDurationContext";
  return NodeType2;
})(NodeType || {});
function validateQuery(query, interpolatedQuery, queryLines, parser) {
  if (!query) {
    return { errors: [], warnings: [] };
  }
  const { errors: interpolatedErrors, warnings: interpolatedWarnings } = parseQuery(interpolatedQuery, parser);
  if (!interpolatedErrors.length && !interpolatedWarnings.length) {
    return { errors: [], warnings: [] };
  }
  let parseErrors = interpolatedErrors;
  let parseWarnings = interpolatedWarnings;
  if (query !== interpolatedQuery) {
    const { errors: queryErrors, warnings: queryWarnings } = parseQuery(query, parser);
    parseErrors = interpolatedErrors.flatMap(
      (interpolatedError) => queryErrors.filter((queryError) => interpolatedError.text === queryError.text) || interpolatedError
    );
    parseWarnings = interpolatedWarnings.flatMap(
      (interpolatedWarning) => queryWarnings.filter((queryWarning) => interpolatedWarning.node.from === queryWarning.node.from) || interpolatedWarning
    );
  }
  const errorBoundaries = parseErrors.map((parseError) => findIssueBoundary(query, queryLines, parseError, "error")).filter(isValidIssueBoundary);
  const warningBoundaries = parseWarnings.map((parseWarning) => findIssueBoundary(query, queryLines, parseWarning, "warning")).filter(isValidIssueBoundary);
  return {
    errors: errorBoundaries,
    warnings: warningBoundaries
  };
}
function parseQuery(query, parser) {
  const parseErrors = [];
  const parseWarnings = [];
  const tree = parser.parse(query);
  tree.iterate({
    enter: (nodeRef) => {
      if (nodeRef.type.id === ErrorId) {
        const node = nodeRef.node;
        parseErrors.push({ node, text: query.substring(node.from, node.to) });
      }
      if (nodeRef.type.name === "SubqueryExpr" /* SubqueryExpr */) {
        const node = nodeRef.node;
        const durations = [];
        const children = node.getChildren("NumberDurationLiteralInDurationContext" /* Duration */);
        for (const child of children) {
          durations.push(query.substring(child.from, child.to));
        }
        if (durations.length === 2 && durations[0] === durations[1]) {
          parseWarnings.push({ node, text: query.substring(node.from, node.to) });
        }
      }
    }
  });
  return { errors: parseErrors, warnings: parseWarnings };
}
function findIssueBoundary(query, queryLines, parseError, issueType) {
  if (queryLines.length === 1) {
    const isEmptyString = parseError.node.from === parseError.node.to;
    const errorNode = isEmptyString && parseError.node.parent ? parseError.node.parent : parseError.node;
    let issue;
    if (issueType === "error") {
      issue = isEmptyString ? query.substring(errorNode.from, errorNode.to) : parseError.text;
    } else {
      issue = warningTypes[parseError.node.type.name];
    }
    return {
      startLineNumber: 1,
      startColumn: errorNode.from + 1,
      endLineNumber: 1,
      endColumn: errorNode.to + 1,
      issue
    };
  }
  let startPos = 0, endPos = 0;
  for (let line = 0; line < queryLines.length; line++) {
    endPos = startPos + queryLines[line].length;
    if (parseError.node.from > endPos) {
      startPos += queryLines[line].length + 1;
      continue;
    }
    return {
      startLineNumber: line + 1,
      startColumn: parseError.node.from - startPos + 1,
      endLineNumber: line + 1,
      endColumn: parseError.node.to - startPos + 1,
      issue: issueType === "error" ? parseError.text : warningTypes[parseError.node.type.name]
    };
  }
  return null;
}
function isValidIssueBoundary(boundary) {
  return boundary !== null;
}
const placeHolderScopedVars = {
  __interval: { text: "1s", value: "1s" },
  __rate_interval: { text: "1s", value: "1s" },
  __auto: { text: "1s", value: "1s" },
  __interval_ms: { text: "1000", value: 1e3 },
  __range_ms: { text: "1000", value: 1e3 },
  __range_s: { text: "1", value: 1 },
  __range: { text: "1s", value: "1s" }
};

export { ErrorId, placeHolderScopedVars, validateQuery, warningTypes };
//# sourceMappingURL=validation.mjs.map
