import { jsxs, jsx, Fragment } from 'react/jsx-runtime';
import { cx, css } from '@emotion/css';
import { Trans, t } from '@grafana/i18n';
import { FormField } from '../FormField/FormField.mjs';
import { Icon } from '../Icon/Icon.mjs';
import { Tooltip } from '../Tooltip/Tooltip.mjs';
import { CertificationKey } from './CertificationKey.mjs';

"use strict";
const TLSAuthSettings = ({ dataSourceConfig, onChange }) => {
  const hasTLSCACert = dataSourceConfig.secureJsonFields && dataSourceConfig.secureJsonFields.tlsCACert;
  const hasTLSClientCert = dataSourceConfig.secureJsonFields && dataSourceConfig.secureJsonFields.tlsClientCert;
  const hasTLSClientKey = dataSourceConfig.secureJsonFields && dataSourceConfig.secureJsonFields.tlsClientKey;
  const hasServerName = dataSourceConfig.jsonData && dataSourceConfig.jsonData.serverName;
  const onResetClickFactory = (field) => (event) => {
    event.preventDefault();
    const newSecureJsonFields = { ...dataSourceConfig.secureJsonFields };
    newSecureJsonFields[field] = false;
    onChange({
      ...dataSourceConfig,
      secureJsonFields: newSecureJsonFields
    });
  };
  const onCertificateChangeFactory = (field) => (event) => {
    const newSecureJsonData = { ...dataSourceConfig.secureJsonData };
    newSecureJsonData[field] = event.currentTarget.value;
    onChange({
      ...dataSourceConfig,
      secureJsonData: newSecureJsonData
    });
  };
  const onServerNameLabelChange = (event) => {
    const newJsonData = {
      ...dataSourceConfig.jsonData,
      serverName: event.currentTarget.value
    };
    onChange({
      ...dataSourceConfig,
      jsonData: newJsonData
    });
  };
  const certificateBeginsWith = "-----BEGIN CERTIFICATE-----";
  const privateKeyBeginsWith = "-----BEGIN RSA PRIVATE KEY-----";
  return /* @__PURE__ */ jsxs("div", { className: "gf-form-group", children: [
    /* @__PURE__ */ jsxs(
      "div",
      {
        className: cx(
          "gf-form",
          css({
            alignItems: "baseline"
          })
        ),
        children: [
          /* @__PURE__ */ jsx("h6", { children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-ui.data-source-settings.tls-heading", children: "TLS/SSL Auth Details" }) }),
          /* @__PURE__ */ jsx(
            Tooltip,
            {
              placement: "right-end",
              content: t(
                "grafana-ui.data-source-settings.tls-tooltip",
                "TLS/SSL Certs are encrypted and stored in the Grafana database."
              ),
              theme: "info",
              children: /* @__PURE__ */ jsx(Icon, { name: "info-circle", size: "xs", style: { marginLeft: "10px" } })
            }
          )
        ]
      }
    ),
    /* @__PURE__ */ jsxs("div", { children: [
      dataSourceConfig.jsonData.tlsAuthWithCACert && /* @__PURE__ */ jsx(
        CertificationKey,
        {
          hasCert: !!hasTLSCACert,
          onChange: onCertificateChangeFactory("tlsCACert"),
          placeholder: t(
            "grafana-ui.data-source-settings.tls-certification-placeholder",
            "Begins with {{certificateBeginsWith}}",
            { certificateBeginsWith }
          ),
          label: t("grafana-ui.data-source-settings.tls-certification-label", "CA Cert"),
          onClick: onResetClickFactory("tlsCACert")
        }
      ),
      dataSourceConfig.jsonData.tlsAuth && /* @__PURE__ */ jsxs(Fragment, { children: [
        /* @__PURE__ */ jsx("div", { className: "gf-form", children: /* @__PURE__ */ jsx(
          FormField,
          {
            label: t("grafana-ui.data-source-settings.tls-server-name-label", "ServerName"),
            labelWidth: 7,
            inputWidth: 30,
            placeholder: "domain.example.com",
            value: hasServerName && dataSourceConfig.jsonData.serverName,
            onChange: onServerNameLabelChange
          }
        ) }),
        /* @__PURE__ */ jsx(
          CertificationKey,
          {
            hasCert: !!hasTLSClientCert,
            label: t("grafana-ui.data-source-settings.tls-client-certification-label", "Client Cert"),
            onChange: onCertificateChangeFactory("tlsClientCert"),
            placeholder: t(
              "grafana-ui.data-source-settings.tls-certification-placeholder",
              "Begins with {{certificateBeginsWith}}",
              { certificateBeginsWith }
            ),
            onClick: onResetClickFactory("tlsClientCert")
          }
        ),
        /* @__PURE__ */ jsx(
          CertificationKey,
          {
            hasCert: !!hasTLSClientKey,
            label: t("grafana-ui.data-source-settings.tls-client-key-label", "Client Key"),
            placeholder: t(
              "grafana-ui.data-source-settings.tls-client-key-placeholder",
              "Begins with {{privateKeyBeginsWith}}",
              { privateKeyBeginsWith }
            ),
            onChange: onCertificateChangeFactory("tlsClientKey"),
            onClick: onResetClickFactory("tlsClientKey")
          }
        )
      ] })
    ] })
  ] });
};

export { TLSAuthSettings };
//# sourceMappingURL=TLSAuthSettings.mjs.map
