import { Property } from 'csstype';
import { SortColumn } from 'react-data-grid';
import { FieldType, Field, GrafanaTheme2, DisplayValue, LinkModel, DisplayValueAlignmentFactors, DataFrame } from '@grafana/data';
import { TableCellDisplayMode, TableCellHeight } from '@grafana/schema';
import { TableCellOptions } from '../types';
import { CellColors, TableRow, ColumnTypes, Comparator } from './types';
export type CellHeightCalculator = (text: string, cellWidth: number) => number;
/**
 * @internal
 * Returns the default row height based on the theme and cell height setting.
 */
export declare function getDefaultRowHeight(theme: GrafanaTheme2, cellHeight?: TableCellHeight): number;
/**
 * @internal
 * Returns true if cell inspection (hover to see full content) is enabled for the field.
 */
export declare function isCellInspectEnabled(field: Field): boolean;
/**
 * @internal
 * Returns true if text wrapping should be applied to the cell.
 */
export declare function shouldTextWrap(field: Field): boolean;
export interface GetMaxWrapCellOptions {
    colWidths: number[];
    avgCharWidth: number;
    wrappedColIdxs: boolean[];
}
/**
 * @internal
 * loop through the fields and their values, determine which cell is going to determine the
 * height of the row based on its content and width, and then return the text, index, and number of lines for that cell.
 */
export declare function getMaxWrapCell(fields: Field[], rowIdx: number, { colWidths, avgCharWidth, wrappedColIdxs }: GetMaxWrapCellOptions): {
    text: string;
    idx: number;
    numLines: number;
};
/**
 * @internal
 * Returns true if text overflow handling should be applied to the cell.
 */
export declare function shouldTextOverflow(field: Field): boolean;
/**
 * @internal
 * Returns the text alignment for a field based on its type and configuration.
 */
export declare function getTextAlign(field?: Field): Property.JustifyContent;
/**
 * @internal
 * Returns the cell options for a field, migrating from legacy displayMode if necessary.
 * TODO: remove live migration in favor of doing it in dashboard or panel migrator
 */
export declare function getCellOptions(field: Field): TableCellOptions;
/**
 * @internal
 * Getting gauge or sparkline values to align is very tricky without looking at all values and passing them through display processor.
 * For very large tables that could pretty expensive. So this is kind of a compromise. We look at the first 1000 rows and cache the longest value.
 * If we have a cached value we just check if the current value is longer and update the alignmentFactor. This can obviously still lead to
 * unaligned gauges but it should a lot less common.
 **/
export declare function getAlignmentFactor(field: Field, displayValue: DisplayValue, rowIndex: number): DisplayValueAlignmentFactors;
/**
 * @internal
 * Returns the text and background colors for a table cell based on its options and display value.
 */
export declare function getCellColors(theme: GrafanaTheme2, cellOptions: TableCellOptions, displayValue: DisplayValue): CellColors;
/**
 * @internal
 * Extracts numeric pixel value from theme spacing
 */
export declare const extractPixelValue: (spacing: string | number) => number;
/**
 * @internal
 */
export declare const getCellLinks: (field: Field, rowIdx: number) => LinkModel<unknown>[] | undefined;
/**
 * @internal
 */
export declare function applySort(rows: TableRow[], fields: Field[], sortColumns: SortColumn[], columnTypes?: ColumnTypes, hasNestedFrames?: boolean): TableRow[];
/**
 * @internal
 */
export declare const frameToRecords: (frame: DataFrame) => TableRow[];
/**
 * @internal
 */
export declare function getComparator(sortColumnType: FieldType): Comparator;
/**
 * Migrates table cell display mode to new object format.
 *
 * @param displayMode The display mode of the cell
 * @returns TableCellOptions object in the correct format
 * relative to the old display mode.
 */
export declare function migrateTableDisplayModeToCellOptions(displayMode: TableCellDisplayMode): TableCellOptions;
/**
 * @internal
 * Returns true if the DataFrame contains nested frames
 */
export declare const getIsNestedTable: (fields: Field[]) => boolean;
/**
 * @internal
 * Processes nested table rows
 */
export declare const processNestedTableRows: (rows: TableRow[], processParents: (parents: TableRow[]) => TableRow[]) => TableRow[];
/**
 * @internal
 * returns the display name of a field
 */
export declare const getDisplayName: (field: Field) => string;
/**
 * @internal
 * returns only fields that are not nested tables and not explicitly hidden
 */
export declare function getVisibleFields(fields: Field[]): Field[];
/**
 * @internal
 * returns a map of column types by display name
 */
export declare function getColumnTypes(fields: Field[]): ColumnTypes;
/**
 * @internal
 * calculates the width of each field, with the following logic:
 * 1. manual sizing minWidth is hard-coded to 50px, we set this in RDG since it enforces the hard limit correctly
 * 2. if minWidth is configured in fieldConfig (or defaults to 150), it serves as the bottom of the auto-size clamp
 */
export declare function computeColWidths(fields: Field[], availWidth: number): number[];
/**
 * @internal
 * if applyToRow is true in any field, return a function that gets the row background color
 */
export declare function getApplyToRowBgFn(fields: Field[], theme: GrafanaTheme2): ((rowIndex: number) => CellColors) | void;
/** @internal */
export declare function withDataLinksActionsTooltip(field: Field, cellType: TableCellDisplayMode): boolean;
