import { jsxs, Fragment, jsx } from 'react/jsx-runtime';
import { css } from '@emotion/css';
import Highlighter from 'react-highlight-words';
import { Trans } from '@grafana/i18n';
import { useTheme2, Tooltip, Icon, Button } from '@grafana/ui';
import { docsTip } from '../../../configuration/shared/utils.mjs';
import { tracking } from './state/helpers.mjs';

"use strict";
function ResultsTable(props) {
  const { metrics, onChange, onClose, query, state, disableTextWrap } = props;
  const theme = useTheme2();
  const styles = getStyles(theme, disableTextWrap);
  function selectMetric(metric) {
    if (metric.value) {
      onChange({ ...query, metric: metric.value });
      tracking("grafana_prom_metric_encycopedia_tracking", state, metric.value);
      onClose();
    }
  }
  function metaRows(metric) {
    var _a, _b, _c, _d;
    if (state.fullMetaSearch && metric) {
      return /* @__PURE__ */ jsxs(Fragment, { children: [
        /* @__PURE__ */ jsx("td", { children: displayType((_a = metric.type) != null ? _a : "") }),
        /* @__PURE__ */ jsx("td", { children: /* @__PURE__ */ jsx(
          Highlighter,
          {
            textToHighlight: (_b = metric.description) != null ? _b : "",
            searchWords: state.metaHaystackMatches,
            autoEscape: true,
            highlightClassName: styles.matchHighLight
          }
        ) })
      ] });
    } else {
      return /* @__PURE__ */ jsxs(Fragment, { children: [
        /* @__PURE__ */ jsx("td", { children: displayType((_c = metric.type) != null ? _c : "") }),
        /* @__PURE__ */ jsx("td", { children: (_d = metric.description) != null ? _d : "" })
      ] });
    }
  }
  function addHelpIcon(fullType, descriptiveType, link) {
    return /* @__PURE__ */ jsxs(Fragment, { children: [
      fullType,
      /* @__PURE__ */ jsx("span", { className: styles.tooltipSpace, children: /* @__PURE__ */ jsx(
        Tooltip,
        {
          content: /* @__PURE__ */ jsxs(Fragment, { children: [
            /* @__PURE__ */ jsxs(Trans, { i18nKey: "grafana-prometheus.querybuilder.results-table.content-descriptive-type", children: [
              "When creating a ",
              { descriptiveType },
              ", Prometheus exposes multiple series with the type counter.",
              " "
            ] }),
            docsTip(link)
          ] }),
          placement: "bottom-start",
          interactive: true,
          children: /* @__PURE__ */ jsx(Icon, { name: "info-circle", size: "xs" })
        }
      ) })
    ] });
  }
  function displayType(type) {
    if (!type) {
      return "";
    }
    if (type.includes("(summary)")) {
      return addHelpIcon(type, "summary", "https://prometheus.io/docs/concepts/metric_types/#summary");
    }
    if (type.includes("(histogram)")) {
      return addHelpIcon(type, "histogram", "https://prometheus.io/docs/concepts/metric_types/#histogram");
    }
    return type;
  }
  function noMetricsMessages() {
    let message;
    if (!state.fuzzySearchQuery) {
      message = "There are no metrics found in the data source.";
    }
    if (query.labels.length > 0) {
      message = "There are no metrics found. Try to expand your label filters.";
    }
    if (state.fuzzySearchQuery || state.selectedTypes.length > 0) {
      message = "There are no metrics found. Try to expand your search and filters.";
    }
    return /* @__PURE__ */ jsx("tr", { className: styles.noResults, children: /* @__PURE__ */ jsx("td", { colSpan: 3, children: message }) });
  }
  function textHighlight(state2) {
    if (state2.useBackend) {
      return [state2.fuzzySearchQuery];
    } else if (state2.fullMetaSearch) {
      return state2.metaHaystackMatches;
    } else {
      return state2.nameHaystackMatches;
    }
  }
  return /* @__PURE__ */ jsxs("table", { className: styles.table, children: [
    /* @__PURE__ */ jsx("thead", { className: styles.stickyHeader, children: /* @__PURE__ */ jsxs("tr", { children: [
      /* @__PURE__ */ jsx("th", { className: `${styles.nameWidth} ${styles.tableHeaderPadding}`, children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.querybuilder.results-table.name", children: "Name" }) }),
      state.hasMetadata && /* @__PURE__ */ jsxs(Fragment, { children: [
        /* @__PURE__ */ jsx("th", { className: `${styles.typeWidth} ${styles.tableHeaderPadding}`, children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.querybuilder.results-table.type", children: "Type" }) }),
        /* @__PURE__ */ jsx("th", { className: `${styles.descriptionWidth} ${styles.tableHeaderPadding}`, children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.querybuilder.results-table.description", children: "Description" }) })
      ] }),
      /* @__PURE__ */ jsx("th", { className: styles.selectButtonWidth, children: " " })
    ] }) }),
    /* @__PURE__ */ jsx("tbody", { children: /* @__PURE__ */ jsxs(Fragment, { children: [
      metrics.length > 0 && metrics.map((metric, idx) => {
        var _a, _b;
        return /* @__PURE__ */ jsxs("tr", { className: styles.row, children: [
          /* @__PURE__ */ jsx("td", { className: styles.nameOverflow, children: /* @__PURE__ */ jsx(
            Highlighter,
            {
              textToHighlight: (_b = metric == null ? void 0 : metric.value) != null ? _b : "",
              searchWords: textHighlight(state),
              autoEscape: true,
              highlightClassName: styles.matchHighLight
            }
          ) }),
          state.hasMetadata && metaRows(metric),
          /* @__PURE__ */ jsx("td", { children: /* @__PURE__ */ jsx(
            Button,
            {
              size: "md",
              variant: "secondary",
              onClick: () => selectMetric(metric),
              className: styles.centerButton,
              children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.querybuilder.results-table.select", children: "Select" })
            }
          ) })
        ] }, (_a = metric == null ? void 0 : metric.value) != null ? _a : idx);
      }),
      metrics.length === 0 && !state.isLoading && noMetricsMessages()
    ] }) })
  ] });
}
const getStyles = (theme, disableTextWrap) => {
  return {
    table: css({
      tableLayout: disableTextWrap ? void 0 : "fixed",
      borderRadius: theme.shape.radius.default,
      width: "100%",
      whiteSpace: disableTextWrap ? "nowrap" : "normal",
      td: {
        padding: theme.spacing(1)
      },
      "td,th": {
        minWidth: theme.spacing(3),
        borderBottom: `1px solid ${theme.colors.border.weak}`
      }
    }),
    row: css({
      label: "row",
      borderBottom: `1px solid ${theme.colors.border.weak}`,
      "&:last-child": {
        borderBottom: 0
      }
    }),
    tableHeaderPadding: css({
      padding: "8px"
    }),
    matchHighLight: css({
      background: "inherit",
      color: theme.components.textHighlight.text,
      backgroundColor: theme.components.textHighlight.background
    }),
    nameWidth: css({
      width: disableTextWrap ? void 0 : "37.5%"
    }),
    nameOverflow: css({
      overflowWrap: disableTextWrap ? void 0 : "anywhere"
    }),
    typeWidth: css({
      width: disableTextWrap ? void 0 : "15%"
    }),
    descriptionWidth: css({
      width: disableTextWrap ? void 0 : "35%"
    }),
    selectButtonWidth: css({
      width: disableTextWrap ? void 0 : "12.5%"
    }),
    stickyHeader: css({
      position: "sticky",
      top: 0,
      backgroundColor: theme.colors.background.primary
    }),
    noResults: css({
      textAlign: "center",
      color: theme.colors.text.secondary
    }),
    tooltipSpace: css({
      marginLeft: "4px"
    }),
    centerButton: css({
      display: "block",
      margin: "auto",
      border: "none"
    })
  };
};

export { ResultsTable };
//# sourceMappingURL=ResultsTable.mjs.map
