import { jsxs, Fragment, jsx } from 'react/jsx-runtime';
import { css } from '@emotion/css';
import { getValueFormat, formattedValueToString } from '@grafana/data';
import { t, Trans } from '@grafana/i18n';
import { useStyles2 } from '../../themes/ThemeContext.mjs';
import { trimFileName } from '../../utils/file.mjs';
import { Button } from '../Button/Button.mjs';
import { Icon } from '../Icon/Icon.mjs';
import { IconButton } from '../IconButton/IconButton.mjs';

"use strict";
const REMOVE_FILE = "Remove file";
function FileListItem({ file: customFile, removeFile }) {
  const styles = useStyles2(getStyles);
  const { file, progress, error, abortUpload, retryUpload } = customFile;
  const renderRightSide = () => {
    if (error) {
      return /* @__PURE__ */ jsxs(Fragment, { children: [
        /* @__PURE__ */ jsx("span", { className: styles.error, children: error.message }),
        retryUpload && /* @__PURE__ */ jsx(
          IconButton,
          {
            name: "sync",
            tooltip: t("grafana-ui.file-dropzone.item-retry", "Retry"),
            tooltipPlacement: "top",
            onClick: retryUpload
          }
        ),
        removeFile && /* @__PURE__ */ jsx(
          IconButton,
          {
            className: retryUpload ? styles.marginLeft : "",
            name: "trash-alt",
            onClick: () => removeFile(customFile),
            tooltip: REMOVE_FILE
          }
        )
      ] });
    }
    if (progress && file.size > progress) {
      return /* @__PURE__ */ jsxs(Fragment, { children: [
        /* @__PURE__ */ jsx("progress", { className: styles.progressBar, max: file.size, value: progress }),
        /* @__PURE__ */ jsxs("span", { className: styles.paddingLeft, children: [
          Math.round(progress / file.size * 100),
          "%"
        ] }),
        abortUpload && /* @__PURE__ */ jsx(Button, { variant: "secondary", type: "button", fill: "text", onClick: abortUpload, children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-ui.file-dropzone.cancel-upload", children: "Cancel upload" }) })
      ] });
    }
    return removeFile && /* @__PURE__ */ jsx(
      IconButton,
      {
        name: "trash-alt",
        onClick: () => removeFile(customFile),
        tooltip: REMOVE_FILE,
        tooltipPlacement: "top"
      }
    );
  };
  const valueFormat = getValueFormat("decbytes")(file.size);
  return /* @__PURE__ */ jsxs("div", { className: styles.fileListContainer, children: [
    /* @__PURE__ */ jsxs("span", { className: styles.fileNameWrapper, children: [
      /* @__PURE__ */ jsx(Icon, { name: "file-blank", size: "lg", "aria-hidden": true }),
      /* @__PURE__ */ jsx("span", { className: styles.padding, children: trimFileName(file.name) }),
      /* @__PURE__ */ jsx("span", { children: formattedValueToString(valueFormat) })
    ] }),
    /* @__PURE__ */ jsx("div", { className: styles.fileNameWrapper, children: renderRightSide() })
  ] });
}
function getStyles(theme) {
  return {
    fileListContainer: css({
      width: "100%",
      display: "flex",
      flexDirection: "row",
      alignItems: "center",
      justifyContent: "space-between",
      padding: theme.spacing(2),
      border: `1px dashed ${theme.colors.border.medium}`,
      backgroundColor: `${theme.colors.background.secondary}`,
      marginTop: theme.spacing(1)
    }),
    fileNameWrapper: css({
      display: "flex",
      flexDirection: "row",
      alignItems: "center"
    }),
    padding: css({
      padding: theme.spacing(0, 1)
    }),
    paddingLeft: css({
      paddingLeft: theme.spacing(2)
    }),
    marginLeft: css({
      marginLeft: theme.spacing(1)
    }),
    error: css({
      paddingRight: theme.spacing(2),
      color: theme.colors.error.text
    }),
    progressBar: css({
      borderRadius: theme.shape.radius.default,
      height: "4px",
      "::-webkit-progress-bar": {
        backgroundColor: theme.colors.border.weak,
        borderRadius: theme.shape.radius.default
      },
      "::-webkit-progress-value": {
        backgroundColor: theme.colors.primary.main,
        borderRadius: theme.shape.radius.default
      }
    })
  };
}

export { FileListItem, REMOVE_FILE };
//# sourceMappingURL=FileListItem.mjs.map
