"use strict";
class Store {
  constructor() {
    this.subscribers = /* @__PURE__ */ new Map();
    window.addEventListener("storage", (e) => {
      if (e.key) {
        this.notifySubscribers(e.key);
      }
    });
  }
  notifySubscribers(key) {
    const keySubscribers = this.subscribers.get(key);
    if (keySubscribers) {
      keySubscribers.forEach((subscriber) => subscriber());
    }
  }
  subscribe(key, callback) {
    if (!this.subscribers.has(key)) {
      this.subscribers.set(key, /* @__PURE__ */ new Set());
    }
    this.subscribers.get(key).add(callback);
    return () => {
      const keySubscribers = this.subscribers.get(key);
      if (keySubscribers) {
        keySubscribers.delete(callback);
        if (keySubscribers.size === 0) {
          this.subscribers.delete(key);
        }
      }
    };
  }
  get(key) {
    return window.localStorage[key];
  }
  set(key, value) {
    window.localStorage[key] = value;
    this.notifySubscribers(key);
  }
  getBool(key, def) {
    if (def !== void 0 && !this.exists(key)) {
      return def;
    }
    return window.localStorage[key] === "true";
  }
  getObject(key, def) {
    let ret = def;
    if (this.exists(key)) {
      const json = window.localStorage[key];
      try {
        ret = JSON.parse(json);
      } catch (error) {
        console.error(`Error parsing store object: ${key}. Returning default: ${def}. [${error}]`);
      }
    }
    return ret;
  }
  /* Returns true when successfully stored, throws error if not successfully stored */
  setObject(key, value) {
    let json;
    try {
      json = JSON.stringify(value);
    } catch (error) {
      throw new Error(`Could not stringify object: ${key}. [${error}]`);
    }
    try {
      this.set(key, json);
    } catch (error) {
      const errorToThrow = new Error(`Could not save item in localStorage: ${key}. [${error}]`);
      if (error instanceof Error) {
        errorToThrow.name = error.name;
      }
      throw errorToThrow;
    }
    return true;
  }
  exists(key) {
    return window.localStorage[key] !== void 0;
  }
  delete(key) {
    window.localStorage.removeItem(key);
    this.notifySubscribers(key);
  }
}
const store = new Store();

export { Store, store };
//# sourceMappingURL=store.mjs.map
