import { jsxs, jsx } from 'react/jsx-runtime';
import { cx, css } from '@emotion/css';
import { isString } from 'lodash';
import { useState, useId, useCallback } from 'react';
import { getTimeZoneInfo } from '@grafana/data';
import { selectors } from '@grafana/e2e-selectors';
import { t, Trans } from '@grafana/i18n';
import { useStyles2 } from '../../../themes/ThemeContext.mjs';
import { Button } from '../../Button/Button.mjs';
import { Combobox } from '../../Combobox/Combobox.mjs';
import { Field } from '../../Forms/Field.mjs';
import { Tab } from '../../Tabs/Tab.mjs';
import { TabContent } from '../../Tabs/TabContent.mjs';
import { TabsBar } from '../../Tabs/TabsBar.mjs';
import { TimeZonePicker } from '../TimeZonePicker.mjs';
import { TimeZoneDescription } from '../TimeZonePicker/TimeZoneDescription.mjs';
import { TimeZoneOffset } from '../TimeZonePicker/TimeZoneOffset.mjs';
import { TimeZoneTitle } from '../TimeZonePicker/TimeZoneTitle.mjs';
import { monthOptions } from '../options.mjs';

"use strict";
const TimePickerFooter = (props) => {
  const {
    timeZone,
    fiscalYearStartMonth,
    timestamp = Date.now(),
    onChangeTimeZone,
    onChangeFiscalYearStartMonth
  } = props;
  const [isEditing, setEditing] = useState(false);
  const [editMode, setEditMode] = useState("tz");
  const timeSettingsId = useId();
  const timeZoneSettingsId = useId();
  const fiscalYearSettingsId = useId();
  const onToggleChangeTimeSettings = useCallback(
    (event) => {
      if (event) {
        event.stopPropagation();
      }
      setEditing(!isEditing);
    },
    [isEditing, setEditing]
  );
  const style = useStyles2(getStyle);
  if (!isString(timeZone)) {
    return null;
  }
  const info = getTimeZoneInfo(timeZone, timestamp);
  if (!info) {
    return null;
  }
  return /* @__PURE__ */ jsxs("div", { children: [
    /* @__PURE__ */ jsxs(
      "section",
      {
        "aria-label": t("time-picker.footer.time-zone-selection", "Time zone selection"),
        className: style.container,
        children: [
          /* @__PURE__ */ jsxs("div", { className: style.timeZoneContainer, children: [
            /* @__PURE__ */ jsxs("div", { className: style.timeZone, children: [
              /* @__PURE__ */ jsx(TimeZoneTitle, { title: info.name }),
              /* @__PURE__ */ jsx("div", { className: style.spacer }),
              /* @__PURE__ */ jsx(TimeZoneDescription, { info })
            ] }),
            /* @__PURE__ */ jsx(TimeZoneOffset, { timeZone, timestamp })
          ] }),
          /* @__PURE__ */ jsx("div", { className: style.spacer }),
          /* @__PURE__ */ jsx(
            Button,
            {
              "data-testid": selectors.components.TimeZonePicker.changeTimeSettingsButton,
              variant: "secondary",
              onClick: onToggleChangeTimeSettings,
              size: "sm",
              "aria-expanded": isEditing,
              "aria-controls": timeSettingsId,
              icon: isEditing ? "angle-up" : "angle-down",
              children: /* @__PURE__ */ jsx(Trans, { i18nKey: "time-picker.footer.change-settings-button", children: "Change time settings" })
            }
          )
        ]
      }
    ),
    isEditing ? /* @__PURE__ */ jsxs("div", { className: style.editContainer, id: timeSettingsId, children: [
      /* @__PURE__ */ jsxs(TabsBar, { children: [
        /* @__PURE__ */ jsx(
          Tab,
          {
            label: t("time-picker.footer.time-zone-option", "Time zone"),
            active: editMode === "tz",
            onChangeTab: () => {
              setEditMode("tz");
            },
            "aria-controls": timeZoneSettingsId
          }
        ),
        /* @__PURE__ */ jsx(
          Tab,
          {
            label: t("time-picker.footer.fiscal-year-option", "Fiscal year"),
            active: editMode === "fy",
            onChangeTab: () => {
              setEditMode("fy");
            },
            "aria-controls": fiscalYearSettingsId
          }
        )
      ] }),
      /* @__PURE__ */ jsx(TabContent, { className: style.noBackground, children: editMode === "tz" ? /* @__PURE__ */ jsx(
        "section",
        {
          role: "tabpanel",
          "data-testid": selectors.components.TimeZonePicker.containerV2,
          id: timeZoneSettingsId,
          className: cx(style.timeZoneContainer, style.timeSettingContainer),
          children: /* @__PURE__ */ jsx(
            TimeZonePicker,
            {
              includeInternal: true,
              onChange: (timeZone2) => {
                onToggleChangeTimeSettings();
                if (isString(timeZone2)) {
                  onChangeTimeZone(timeZone2);
                }
              },
              onBlur: onToggleChangeTimeSettings,
              menuShouldPortal: false
            }
          )
        }
      ) : /* @__PURE__ */ jsx(
        "section",
        {
          role: "tabpanel",
          "data-testid": selectors.components.TimeZonePicker.containerV2,
          id: fiscalYearSettingsId,
          className: cx(style.timeZoneContainer, style.timeSettingContainer),
          children: /* @__PURE__ */ jsx(
            Field,
            {
              className: style.fiscalYearField,
              label: t("time-picker.footer.fiscal-year-start", "Fiscal year start month"),
              children: /* @__PURE__ */ jsx(
                Combobox,
                {
                  value: fiscalYearStartMonth != null ? fiscalYearStartMonth : null,
                  options: monthOptions,
                  onChange: (value) => {
                    var _a;
                    if (onChangeFiscalYearStartMonth) {
                      onChangeFiscalYearStartMonth((_a = value == null ? void 0 : value.value) != null ? _a : 0);
                    }
                  }
                }
              )
            }
          )
        }
      ) })
    ] }) : null
  ] });
};
const getStyle = (theme) => ({
  container: css({
    borderTop: `1px solid ${theme.colors.border.weak}`,
    padding: theme.spacing(1.5),
    display: "flex",
    flexDirection: "row",
    justifyContent: "space-between",
    alignItems: "center"
  }),
  editContainer: css({
    borderTop: `1px solid ${theme.colors.border.weak}`,
    padding: theme.spacing(1.5),
    paddingTop: 0,
    justifyContent: "space-between",
    alignItems: "center"
  }),
  spacer: css({
    marginLeft: "7px"
  }),
  timeSettingContainer: css({
    paddingTop: theme.spacing(1)
  }),
  noBackground: css({
    background: "inherit"
  }),
  fiscalYearField: css({
    marginBottom: 0
  }),
  timeZoneContainer: css({
    display: "flex",
    flexDirection: "row",
    justifyContent: "space-between",
    alignItems: "center",
    flexGrow: 1
  }),
  timeZone: css({
    display: "flex",
    flexDirection: "row",
    alignItems: "baseline",
    flexGrow: 1
  })
});

export { TimePickerFooter };
//# sourceMappingURL=TimePickerFooter.mjs.map
