import { map } from 'lodash';
import { METRIC_LABEL } from './constants.mjs';
import { getPrometheusTime } from './language_utils.mjs';
import { PrometheusLabelNamesRegex, PrometheusLabelNamesRegexWithMatch, PrometheusLabelValuesRegex, PrometheusMetricNamesRegex, PrometheusQueryResultRegex } from './migrations/variableMigration.mjs';
import { getOriginalMetricName } from './result_transformer.mjs';

"use strict";
class PrometheusMetricFindQuery {
  constructor(datasource, query) {
    this.datasource = datasource;
    this.query = query;
    this.datasource = datasource;
    this.query = query;
  }
  async process(timeRange) {
    const labelNamesRegex = PrometheusLabelNamesRegex;
    const labelNamesRegexWithMatch = PrometheusLabelNamesRegexWithMatch;
    const labelValuesRegex = PrometheusLabelValuesRegex;
    const metricNamesRegex = PrometheusMetricNamesRegex;
    const queryResultRegex = PrometheusQueryResultRegex;
    const labelNamesQuery = this.query.match(labelNamesRegex);
    const labelNamesMatchQuery = this.query.match(labelNamesRegexWithMatch);
    if (labelNamesMatchQuery) {
      const selector = `{__name__=~".*${labelNamesMatchQuery[1]}.*"}`;
      const keys = await this.datasource.languageProvider.queryLabelKeys(timeRange, selector);
      return keys.filter((key) => key !== METRIC_LABEL).map((result) => ({ text: result }));
    }
    if (labelNamesQuery) {
      return this.datasource.getTagKeys({ filters: [], timeRange });
    }
    const labelValuesQuery = this.query.match(labelValuesRegex);
    if (labelValuesQuery) {
      const filter = labelValuesQuery[1];
      const label = labelValuesQuery[2];
      if (isFilterDefined(filter)) {
        return await this.labelValuesQuery(label, timeRange, filter);
      } else {
        return await this.labelValuesQuery(label, timeRange);
      }
    }
    const metricNamesQuery = this.query.match(metricNamesRegex);
    if (metricNamesQuery) {
      return await this.metricNameQuery(metricNamesQuery[1], timeRange);
    }
    const queryResultQuery = this.query.match(queryResultRegex);
    if (queryResultQuery) {
      return this.queryResultQuery(queryResultQuery[1], timeRange);
    }
    const expressions = ["label_values()", "metrics()", "query_result()"];
    if (!expressions.includes(this.query)) {
      return await this.metricNameAndLabelsQuery(this.query, timeRange);
    }
    return Promise.resolve([]);
  }
  async labelValuesQuery(label, range, metric) {
    const values = await this.datasource.languageProvider.queryLabelValues(range, label, metric);
    return values.map((value) => ({ text: value }));
  }
  async metricNameQuery(metricFilterPattern, range) {
    const names = await this.datasource.languageProvider.queryLabelValues(
      range,
      METRIC_LABEL,
      `{__name__=~"${metricFilterPattern}"}`
    );
    return names.map((n) => ({ text: n, expandable: true }));
  }
  queryResultQuery(query, range) {
    const url = "/api/v1/query";
    const params = {
      query,
      time: getPrometheusTime(range.to, true).toString()
    };
    return this.datasource.metadataRequest(url, params).then((result) => {
      switch (result.data.data.resultType) {
        case "scalar":
        // [ <unix_time>, "<scalar_value>" ]
        case "string":
          return [
            {
              text: result.data.data.result[1] || "",
              expandable: false
            }
          ];
        case "vector":
          return map(result.data.data.result, (metricData) => {
            let text = metricData.metric.__name__ || "";
            delete metricData.metric.__name__;
            text += "{" + map(metricData.metric, (v, k) => {
              return k + '="' + v + '"';
            }).join(",") + "}";
            text += " " + metricData.value[1] + " " + metricData.value[0] * 1e3;
            return {
              text,
              expandable: true
            };
          });
        default:
          throw Error(`Unknown/Unhandled result type: [${result.data.data.resultType}]`);
      }
    });
  }
  async metricNameAndLabelsQuery(query, range) {
    const start = getPrometheusTime(range.from, false);
    const end = getPrometheusTime(range.to, true);
    const params = {
      "match[]": query,
      start: start.toString(),
      end: end.toString()
    };
    const result = await this.datasource.metadataRequest(`/api/v1/series`, params);
    return result.data.data.map((metric) => ({
      text: getOriginalMetricName(metric),
      expandable: true
    }));
  }
}
function isFilterDefined(filter) {
  return filter && filter.split(" ").join("") !== "{}";
}

export { PrometheusMetricFindQuery };
//# sourceMappingURL=metric_find_query.mjs.map
