import { jsxs, jsx } from 'react/jsx-runtime';
import { css } from '@emotion/css';
import { capitalize } from 'lodash';
import { useState, useMemo } from 'react';
import { getNextRefId } from '@grafana/data';
import { t, Trans } from '@grafana/i18n';
import { reportInteraction } from '@grafana/runtime';
import { useStyles2, Modal, Collapse, Button } from '@grafana/ui';
import { QueryPattern } from './QueryPattern.mjs';
import { buildVisualQueryFromString } from './parsing.mjs';
import { promQueryModeller } from './shared/modeller_instance.mjs';
import { PromQueryPatternType } from './types.mjs';

"use strict";
const QueryPatternsModal = (props) => {
  const { isOpen, onClose, onChange, onAddQuery, query, queries, app } = props;
  const [openTabs, setOpenTabs] = useState([]);
  const [selectedPatternName, setSelectedPatternName] = useState(null);
  const styles = useStyles2(getStyles);
  const hasNewQueryOption = !!onAddQuery;
  const hasPreviousQuery = useMemo(() => {
    var _a;
    const visualQuery = buildVisualQueryFromString((_a = query.expr) != null ? _a : "");
    const hasOperations = visualQuery.query.operations.length > 0, hasMetric = visualQuery.query.metric, hasLabels = visualQuery.query.labels.length > 0, hasBinaryQueries = visualQuery.query.binaryQueries ? visualQuery.query.binaryQueries.length > 0 : false;
    return hasOperations || hasMetric || hasLabels || hasBinaryQueries;
  }, [query.expr]);
  const onPatternSelect = (pattern, selectAsNewQuery = false) => {
    const visualQuery = buildVisualQueryFromString(selectAsNewQuery ? "" : query.expr);
    reportInteraction("grafana_prom_kickstart_your_query_selected", {
      app: app != null ? app : "",
      editorMode: query.editorMode,
      selectedPattern: pattern.name,
      preSelectedOperationsCount: visualQuery.query.operations.length,
      preSelectedLabelsCount: visualQuery.query.labels.length,
      createNewQuery: hasNewQueryOption && selectAsNewQuery
    });
    visualQuery.query.operations = pattern.operations;
    visualQuery.query.binaryQueries = pattern.binaryQueries;
    const renderedExpr = promQueryModeller.renderQuery(visualQuery.query);
    if (hasNewQueryOption && selectAsNewQuery) {
      onAddQuery({
        ...query,
        refId: getNextRefId(queries != null ? queries : [query]),
        expr: renderedExpr
      });
    } else {
      onChange({
        ...query,
        expr: renderedExpr
      });
    }
    setSelectedPatternName(null);
    onClose();
  };
  return /* @__PURE__ */ jsxs(
    Modal,
    {
      "aria-label": t(
        "grafana-prometheus.querybuilder.query-patterns-modal.aria-label-kick-start-your-query-modal",
        "Kick start your query modal"
      ),
      isOpen,
      title: t(
        "grafana-prometheus.querybuilder.query-patterns-modal.title-kick-start-your-query",
        "Kick start your query"
      ),
      onDismiss: onClose,
      children: [
        /* @__PURE__ */ jsx("div", { className: styles.spacing, children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.querybuilder.query-patterns-modal.description-kick-start-your-query", children: "Kick start your query by selecting one of these queries. You can then continue to complete your query." }) }),
        Object.values(PromQueryPatternType).map((patternType) => {
          const isOpen2 = openTabs.includes(patternType);
          return /* @__PURE__ */ jsx(
            Collapse,
            {
              "aria-label": t(
                "grafana-prometheus.querybuilder.query-patterns-modal.aria-label-toggle-query-starter",
                "open and close {{patternType}} query starter card",
                { patternType }
              ),
              label: t(
                "grafana-prometheus.querybuilder.query-patterns-modal.label-toggle-query-starter",
                "{{patternType}} query starters",
                {
                  patternType: capitalize(patternType)
                }
              ),
              isOpen: isOpen2,
              collapsible: true,
              onToggle: () => {
                const action = isOpen2 ? "close" : "open";
                reportInteraction(`grafana_prom_kickstart_toggle_pattern_card`, {
                  action,
                  patternType
                });
                setOpenTabs(
                  (tabs) => (
                    // close tab if it's already open, otherwise open it
                    tabs.includes(patternType) ? tabs.filter((t2) => t2 !== patternType) : [...tabs, patternType]
                  )
                );
              },
              children: /* @__PURE__ */ jsx("div", { className: styles.cardsContainer, children: promQueryModeller.getQueryPatterns().filter((pattern) => pattern.type === patternType).map((pattern) => /* @__PURE__ */ jsx(
                QueryPattern,
                {
                  pattern,
                  hasNewQueryOption,
                  hasPreviousQuery,
                  onPatternSelect,
                  selectedPatternName,
                  setSelectedPatternName
                },
                pattern.name
              )) })
            },
            patternType
          );
        }),
        /* @__PURE__ */ jsx(
          Button,
          {
            "aria-label": t(
              "grafana-prometheus.querybuilder.query-patterns-modal.aria-label-close-kick-start-your-query-modal",
              "close kick start your query modal"
            ),
            variant: "secondary",
            onClick: onClose,
            children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.querybuilder.query-patterns-modal.close", children: "Close" })
          }
        )
      ]
    }
  );
};
const getStyles = (theme) => {
  return {
    cardsContainer: css({
      display: "flex",
      flexDirection: "row",
      flexWrap: "wrap",
      justifyContent: "space-between"
    }),
    spacing: css({
      marginBottom: theme.spacing(1)
    })
  };
};

export { QueryPatternsModal };
//# sourceMappingURL=QueryPatternsModal.mjs.map
