import { invert } from 'lodash';
import { Token } from 'prismjs';
import { AbstractLabelOperator, incrRoundDn, dateMath } from '@grafana/data';
import { addLabelToQuery } from './add_label_to_query.mjs';
import { getCacheDurationInMinutes } from './caching.mjs';
import { SUGGESTIONS_LIMIT, PROMETHEUS_QUERY_BUILDER_MAX_RESULTS } from './constants.mjs';
import { PrometheusCacheLevel } from './types.mjs';

"use strict";
const processHistogramMetrics = (metrics) => {
  const resultSet = /* @__PURE__ */ new Set();
  const regexp = new RegExp("_bucket($|:)");
  for (let index = 0; index < metrics.length; index++) {
    const metric = metrics[index];
    const isHistogramValue = regexp.test(metric);
    if (isHistogramValue) {
      resultSet.add(metric);
    }
  }
  return [...resultSet];
};
function processLabels(labels, withName = false) {
  const valueSet = {};
  labels.forEach((label) => {
    const { __name__, ...rest } = label;
    if (withName) {
      valueSet["__name__"] = valueSet["__name__"] || /* @__PURE__ */ new Set();
      if (!valueSet["__name__"].has(__name__)) {
        valueSet["__name__"].add(__name__);
      }
    }
    Object.keys(rest).forEach((key) => {
      if (!valueSet[key]) {
        valueSet[key] = /* @__PURE__ */ new Set();
      }
      if (!valueSet[key].has(rest[key])) {
        valueSet[key].add(rest[key]);
      }
    });
  });
  const valueArray = {};
  limitSuggestions(Object.keys(valueSet)).forEach((key) => {
    valueArray[key] = limitSuggestions(Array.from(valueSet[key]));
  });
  return { values: valueArray, keys: Object.keys(valueArray) };
}
const selectorRegexp = /\{[^}]*?(\}|$)/;
const labelRegexp = /\b(\w+)(!?=~?)("[^"\n]*?")(,)?(\s*)?/g;
function parseSelector(query, cursorOffset = 1) {
  if (!query.match(selectorRegexp)) {
    if (query.match(/^[A-Za-z:][\w:]*$/)) {
      return {
        selector: `{__name__="${query}"}`,
        labelKeys: ["__name__"]
      };
    }
    throw new Error("Query must contain a selector: " + query);
  }
  const prefix = query.slice(0, cursorOffset);
  const prefixOpen = prefix.lastIndexOf("{");
  const prefixClose = prefix.lastIndexOf("}");
  if (prefixOpen === -1) {
    throw new Error("Not inside selector, missing open brace: " + prefix);
  }
  if (prefixClose > -1 && prefixClose > prefixOpen) {
    throw new Error("Not inside selector, previous selector already closed: " + prefix);
  }
  const suffix = query.slice(cursorOffset);
  const suffixCloseIndex = suffix.indexOf("}");
  const suffixClose = suffixCloseIndex + cursorOffset;
  const suffixOpenIndex = suffix.indexOf("{");
  const suffixOpen = suffixOpenIndex + cursorOffset;
  if (suffixClose === -1) {
    throw new Error("Not inside selector, missing closing brace in suffix: " + suffix);
  }
  if (suffixOpenIndex > -1 && suffixOpen < suffixClose) {
    throw new Error("Not inside selector, next selector opens before this one closed: " + suffix);
  }
  const selector = query.slice(prefixOpen, suffixClose);
  const labels = {};
  selector.replace(labelRegexp, (label, key, operator, value) => {
    const labelOffset = query.indexOf(label);
    const valueStart = labelOffset + key.length + operator.length + 1;
    const valueEnd = labelOffset + key.length + operator.length + value.length - 1;
    if (cursorOffset < valueStart || cursorOffset > valueEnd) {
      labels[key] = { value, operator };
    }
    return "";
  });
  const metricPrefix = query.slice(0, prefixOpen);
  const metricMatch = metricPrefix.match(/[A-Za-z:][\w:]*$/);
  if (metricMatch) {
    labels["__name__"] = { value: `"${metricMatch[0]}"`, operator: "=" };
  }
  const labelKeys = Object.keys(labels).sort();
  const cleanSelector = labelKeys.map((key) => `${key}${labels[key].operator}${labels[key].value}`).join(",");
  const selectorString = ["{", cleanSelector, "}"].join("");
  return { labelKeys, selector: selectorString };
}
function expandRecordingRules(query, mapping) {
  const getRuleRegex = (ruleName) => new RegExp(`(\\s|\\(|^)(${ruleName})(\\s|$|\\(|\\[|\\{)`, "ig");
  const tmpSplitParts = Object.keys(mapping).reduce(
    (prev, curr) => {
      let parts = [];
      let tmpParts = [];
      let removeIdx = [];
      prev.filter(Boolean).forEach((p, i) => {
        const doesMatch = p.match(getRuleRegex(curr));
        if (doesMatch) {
          parts = p.split(curr);
          if (parts.length === 2) {
            removeIdx.push(i);
            tmpParts.push(...[parts[0], curr, parts[1]].filter(Boolean));
          } else if (parts.length > 2) {
            removeIdx.push(i);
            parts = parts.map((p2) => p2 === "" ? curr : p2);
            tmpParts.push(...parts);
          }
        }
      });
      removeIdx.forEach((ri) => prev[ri] = "");
      prev = prev.filter(Boolean);
      prev.push(...tmpParts);
      return prev;
    },
    [query]
  );
  let labelFound = false;
  const trulyExpandedQuery = tmpSplitParts.map((tsp, i) => {
    if (labelFound) {
      labelFound = false;
      return "";
    }
    if (mapping[tsp]) {
      const { expandedQuery: recordingRule, identifierValue, identifier } = mapping[tsp];
      if (i + 1 !== tmpSplitParts.length && tmpSplitParts[i + 1].match(labelRegexp)) {
        labelFound = true;
        const regexp = new RegExp(`(,)?(\\s)?(${identifier}=\\"${identifierValue}\\")(,)?(\\s)?`, "g");
        const labels = tmpSplitParts[i + 1].replace(regexp, "");
        const invalidLabelsRegex = /(\)\{|\}\{|\]\{)/;
        return addLabelsToExpression(recordingRule + labels, invalidLabelsRegex);
      } else {
        return recordingRule;
      }
    }
    return tsp;
  });
  return trulyExpandedQuery.filter(Boolean).join("");
}
function addLabelsToExpression(expr, invalidLabelsRegexp) {
  var _a;
  const match = expr.match(invalidLabelsRegexp);
  if (!match) {
    return expr;
  }
  const indexOfRegexMatch = (_a = match.index) != null ? _a : 0;
  const exprBeforeRegexMatch = expr.slice(0, indexOfRegexMatch + 1);
  const exprAfterRegexMatch = expr.slice(indexOfRegexMatch + 1);
  const arrayOfLabelObjects = [];
  exprAfterRegexMatch.replace(labelRegexp, (label, key, operator, value, comma, space) => {
    arrayOfLabelObjects.push({ key, operator, value, comma, space });
    return "";
  });
  let result = exprBeforeRegexMatch;
  arrayOfLabelObjects.filter(Boolean).forEach((obj) => {
    const value = obj.value.slice(1, -1);
    result = addLabelToQuery(result, obj.key, value, obj.operator);
  });
  let existingLabel = arrayOfLabelObjects.reduce((prev, curr) => {
    var _a2, _b;
    prev += `${curr.key}${curr.operator}${curr.value}${(_a2 = curr.comma) != null ? _a2 : ""}${(_b = curr.space) != null ? _b : ""}`;
    return prev;
  }, "");
  existingLabel = "{" + existingLabel + "}";
  const potentialLeftOver = exprAfterRegexMatch.replace(existingLabel, "");
  return result + potentialLeftOver;
}
function fixSummariesMetadata(metadata) {
  if (!metadata) {
    return metadata;
  }
  const baseMetadata = {};
  const summaryMetadata = {};
  for (const metric in metadata) {
    const item = metadata[metric][0];
    baseMetadata[metric] = item;
    if (item.type === "histogram") {
      summaryMetadata[`${metric}_bucket`] = {
        type: "counter",
        help: `Cumulative counters for the observation buckets (${item.help})`
      };
      summaryMetadata[`${metric}_count`] = {
        type: "counter",
        help: `Count of events that have been observed for the histogram metric (${item.help})`
      };
      summaryMetadata[`${metric}_sum`] = {
        type: "counter",
        help: `Total sum of all observed values for the histogram metric (${item.help})`
      };
    }
    if (item.type === "summary") {
      summaryMetadata[`${metric}_count`] = {
        type: "counter",
        help: `Count of events that have been observed for the base metric (${item.help})`
      };
      summaryMetadata[`${metric}_sum`] = {
        type: "counter",
        help: `Total sum of all observed values for the base metric (${item.help})`
      };
    }
  }
  const syntheticMetadata = {};
  syntheticMetadata["ALERTS"] = {
    type: "gauge",
    help: "Time series showing pending and firing alerts. The sample value is set to 1 as long as the alert is in the indicated active (pending or firing) state."
  };
  return { ...baseMetadata, ...summaryMetadata, ...syntheticMetadata };
}
function roundMsToMin(milliseconds) {
  return roundSecToMin(milliseconds / 1e3);
}
function roundSecToMin(seconds) {
  return Math.floor(seconds / 60);
}
function roundSecToNextMin(seconds, secondsToRound = 1) {
  return Math.ceil(seconds / 60) - Math.ceil(seconds / 60) % secondsToRound;
}
function limitSuggestions(items) {
  return items.slice(0, SUGGESTIONS_LIMIT);
}
function addLimitInfo(items) {
  return items && items.length >= SUGGESTIONS_LIMIT ? `, limited to the first ${SUGGESTIONS_LIMIT} received items` : "";
}
const FromPromLikeMap = {
  "=": AbstractLabelOperator.Equal,
  "!=": AbstractLabelOperator.NotEqual,
  "=~": AbstractLabelOperator.EqualRegEx,
  "!~": AbstractLabelOperator.NotEqualRegEx
};
const ToPromLikeMap = invert(FromPromLikeMap);
function toPromLikeExpr(labelBasedQuery) {
  const expr = labelBasedQuery.labelMatchers.map((selector) => {
    const operator = ToPromLikeMap[selector.operator];
    if (operator) {
      return `${selector.name}${operator}"${selector.value}"`;
    } else {
      return "";
    }
  }).filter((e) => e !== "").join(", ");
  return expr ? `{${expr}}` : "";
}
function toPromLikeQuery(labelBasedQuery) {
  return {
    refId: labelBasedQuery.refId,
    expr: toPromLikeExpr(labelBasedQuery),
    range: true
  };
}
function getMaybeTokenStringContent(token) {
  if (typeof token.content === "string") {
    return token.content;
  }
  return "";
}
function extractLabelMatchers(tokens) {
  const labelMatchers = [];
  for (const token of tokens) {
    if (!(token instanceof Token)) {
      continue;
    }
    if (token.type === "context-labels") {
      let labelKey = "";
      let labelValue = "";
      let labelOperator = "";
      const contentTokens = Array.isArray(token.content) ? token.content : [token.content];
      for (let currentToken of contentTokens) {
        if (typeof currentToken === "string") {
          let currentStr;
          currentStr = currentToken;
          if (currentStr === "=" || currentStr === "!=" || currentStr === "=~" || currentStr === "!~") {
            labelOperator = currentStr;
          }
        } else if (currentToken instanceof Token) {
          switch (currentToken.type) {
            case "label-key":
              labelKey = getMaybeTokenStringContent(currentToken);
              break;
            case "label-value":
              labelValue = getMaybeTokenStringContent(currentToken);
              labelValue = labelValue.substring(1, labelValue.length - 1);
              const labelComparator = FromPromLikeMap[labelOperator];
              if (labelComparator) {
                labelMatchers.push({ name: labelKey, operator: labelComparator, value: labelValue });
              }
              break;
          }
        }
      }
    }
  }
  return labelMatchers;
}
function getRangeSnapInterval(cacheLevel, range) {
  if (cacheLevel === PrometheusCacheLevel.None) {
    return {
      start: getPrometheusTime(range.from, false).toString(),
      end: getPrometheusTime(range.to, true).toString()
    };
  }
  const startTime = getPrometheusTime(range.from, false);
  const startTimeQuantizedSeconds = incrRoundDn(startTime, getCacheDurationInMinutes(cacheLevel) * 60);
  const endTime = getPrometheusTime(range.to, true);
  const endTimeQuantizedSeconds = roundSecToNextMin(endTime, getCacheDurationInMinutes(cacheLevel)) * 60;
  if (startTimeQuantizedSeconds === endTimeQuantizedSeconds) {
    const endTimePlusOneStep = endTimeQuantizedSeconds + getCacheDurationInMinutes(cacheLevel) * 60;
    return { start: startTimeQuantizedSeconds.toString(), end: endTimePlusOneStep.toString() };
  }
  const start = startTimeQuantizedSeconds.toString();
  const end = endTimeQuantizedSeconds.toString();
  return { start, end };
}
function getPrometheusTime(date, roundUp) {
  if (typeof date === "string") {
    date = dateMath.parse(date, roundUp);
  }
  return Math.ceil(date.valueOf() / 1e3);
}
function truncateResult(array, limit) {
  if (limit === void 0) {
    limit = PROMETHEUS_QUERY_BUILDER_MAX_RESULTS;
  }
  array.length = Math.min(array.length, limit);
  return array;
}

export { addLimitInfo, expandRecordingRules, extractLabelMatchers, fixSummariesMetadata, getPrometheusTime, getRangeSnapInterval, labelRegexp, limitSuggestions, parseSelector, processHistogramMetrics, processLabels, roundMsToMin, roundSecToMin, roundSecToNextMin, selectorRegexp, toPromLikeExpr, toPromLikeQuery, truncateResult };
//# sourceMappingURL=language_utils.mjs.map
