import { jsx } from 'react/jsx-runtime';
import { css } from '@emotion/css';
import { Trans } from '@grafana/i18n';
import { TextLink, FieldValidationMessage } from '@grafana/ui';

"use strict";
function docsTip(url) {
  const docsUrl = "https://grafana.com/docs/grafana/latest/datasources/prometheus/configure-prometheus-data-source/";
  return /* @__PURE__ */ jsx(TextLink, { href: url ? url : docsUrl, external: true, children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.configuration.docs-tip.visit-docs-for-more-details-here", children: "Visit docs for more details here." }) });
}
const validateInput = (input, pattern, errorMessage) => {
  const defaultErrorMessage = "Value is not valid";
  const inputTooLongErrorMessage = "Input is too long";
  const validationTimeoutErrorMessage = "Validation timeout - input too complex";
  const invalidValidationPatternErrorMessage = "Invalid validation pattern";
  const MAX_INPUT_LENGTH = 1e3;
  if (!input) {
    return true;
  }
  if (input.length > MAX_INPUT_LENGTH) {
    return /* @__PURE__ */ jsx(FieldValidationMessage, { children: inputTooLongErrorMessage });
  }
  try {
    let regex = typeof pattern === "string" ? new RegExp(pattern) : pattern;
    if (typeof pattern === "string" && !pattern.startsWith("^") && !pattern.endsWith("$")) {
      regex = new RegExp(`^${pattern}$`);
    }
    const timeout = 100;
    const startTime = Date.now();
    const isValid = regex.test(input);
    if (Date.now() - startTime > timeout) {
      return /* @__PURE__ */ jsx(FieldValidationMessage, { children: validationTimeoutErrorMessage });
    }
    if (!isValid) {
      return /* @__PURE__ */ jsx(FieldValidationMessage, { children: errorMessage || defaultErrorMessage });
    }
    return true;
  } catch (error) {
    return /* @__PURE__ */ jsx(FieldValidationMessage, { children: invalidValidationPatternErrorMessage });
  }
};
function overhaulStyles(theme) {
  return {
    additionalSettings: css({
      marginBottom: "25px"
    }),
    secondaryGrey: css({
      color: theme.colors.secondary.text,
      opacity: "65%"
    }),
    inlineError: css({
      margin: "0px 0px 4px 245px"
    }),
    switchField: css({
      alignItems: "center"
    }),
    sectionHeaderPadding: css({
      paddingTop: "32px"
    }),
    sectionBottomPadding: css({
      paddingBottom: "28px"
    }),
    subsectionText: css({
      fontSize: "12px"
    }),
    hrBottomSpace: css({
      marginBottom: "56px"
    }),
    hrTopSpace: css({
      marginTop: "50px"
    }),
    textUnderline: css({
      textDecoration: "underline"
    }),
    versionMargin: css({
      marginBottom: "12px"
    }),
    advancedHTTPSettingsMargin: css({
      margin: "24px 0 8px 0"
    }),
    advancedSettings: css({
      paddingTop: "32px"
    }),
    alertingTop: css({
      marginTop: "40px !important"
    }),
    overhaulPageHeading: css({
      fontWeight: 400
    }),
    container: css({
      maxwidth: 578
    })
  };
}

export { docsTip, overhaulStyles, validateInput };
//# sourceMappingURL=utils.mjs.map
